<?php

namespace App\Livewire;

use App\Models\LibraryTrack;
use App\Services\LibraryService;
use Livewire\Component;
use Livewire\Attributes\Title;
use Livewire\Attributes\Layout;

class TrackDetail extends Component
{
    public $slug;
    public $track;
    public $relatedTracks = [];
    public $isPlaying = false;
    public $currentTime = 0;
    public $duration = 0;
    public $volume = 1.0;
    public $isMuted = false;
    public $playbackRate = 1.0;
    public $isDemoMode = false;
    public $audioError = null;
    public $audioLoading = false;
    public $lastPlayedAt = null;

    public function mount($slug)
    {
        $this->slug = $slug;
        $this->loadTrack();
        $this->loadRelatedTracks();
        
        // Set demo duration for tracks without actual audio files
        if ($this->track && !$this->track->duration) {
            $this->track->duration = 180; // 3 minutes default for demo
        }
    }

    public function loadTrack()
    {
        $this->track = LibraryTrack::where('slug', $this->slug)
            ->active()
            ->with('category')
            ->first();

        if (!$this->track) {
            abort(404);
        }
    }

    public function loadRelatedTracks()
    {
        $this->relatedTracks = LibraryTrack::where('category_id', $this->track->category_id)
            ->where('id', '!=', $this->track->id)
            ->active()
            ->limit(4)
            ->get();
    }

    public function play()
    {
        $this->isPlaying = true;
        $this->audioError = null;
        $this->lastPlayedAt = now();
        
        // Increment play count when starting playback
        if ($this->track) {
            $this->track->incrementPlayCount();
        }
        
        $this->dispatch('play-audio', trackId: $this->track->id);
    }

    public function pause()
    {
        $this->isPlaying = false;
        $this->dispatch('pause-audio');
    }

    public function stop()
    {
        $this->isPlaying = false;
        $this->currentTime = 0;
        $this->dispatch('stop-audio');
    }

    public function updateProgress($currentTime, $duration)
    {
        $this->currentTime = $currentTime;
        $this->duration = $duration;
    }

    public function seekTo($time)
    {
        $this->currentTime = $time;
        $this->dispatch('seek-audio', time: $time);
    }

    public function setVolume($volume)
    {
        $this->volume = max(0, min(1, $volume));
        $this->isMuted = $this->volume === 0;
        $this->dispatch('set-volume', volume: $this->volume);
    }

    public function toggleMute()
    {
        $this->isMuted = !$this->isMuted;
        if ($this->isMuted) {
            $this->dispatch('set-volume', volume: 0);
        } else {
            $this->dispatch('set-volume', volume: $this->volume);
        }
    }

    public function setPlaybackRate($rate)
    {
        $this->playbackRate = $rate;
        $this->dispatch('set-playback-rate', rate: $rate);
    }

    public function onAudioLoad()
    {
        $this->audioLoading = false;
        $this->audioError = null;
    }

    public function onAudioError($error)
    {
        $this->audioLoading = false;
        $this->audioError = $error;
        $this->isPlaying = false;
    }

    public function onAudioEnded()
    {
        $this->isPlaying = false;
        $this->currentTime = 0;
        $this->dispatch('audio-ended');
    }

    #[Title('Track Detail - Vybe')]
    public function render()
    {
        return view('livewire.pages.track-detail');
    }
}
