<?php

namespace App\Http\Livewire\Production\Reports;

use App\Models\Seasons;
use Livewire\Component;
use App\Models\Customer;
use App\Models\Shipment;
use App\Models\Suppliers;
use App\Helper\Conversions;
use App\Models\Departments;
use Livewire\Attributes\On;
use App\Models\ShipmentLine;
use Livewire\WithPagination;
use App\Services\PriceFinder;
use App\Models\CustomerOrders;
use App\Models\ShipmentLineSizes;
use Livewire\Attributes\Computed;
use Livewire\Attributes\Reactive;
use Livewire\Attributes\Validate;
use Illuminate\Support\Facades\DB;
use Livewire\Attributes\Renderless;
use Illuminate\Support\Facades\Gate;
use App\Http\Livewire\Traits\Filterable;
use Illuminate\Support\Facades\Validator;
use App\Http\Livewire\FilterableComponent;
use App\Models\CustomerOrderLineQuantities;

class ShipmentSchedule extends FilterableComponent
{
    use WithPagination;
    use Filterable;

    public $hideMabli = 0;

    // Properties for filters and display options
    public $hideFilters = false;
    public $borders = false;
    public $view = 'unshipped';
    public $fromDate = '2023-10-01';
    public $toDate = '2023-12-31';
    public $department = [];
    public $customer = [];
    public $season = [];
    public $factory = [];
    public $truck = "";
    public $noTruck = false;
    public $coo = [];
    public $category = [];
    public $search = "";
    public $columns = [
        'Season' => true,
        'Factory' => true,
        'Customer' => true,
        'Order No' => true,
        'Style' => true,
        'Description' => true,
        'Colour' => true,
        'Image' => true,
        'Sizes' => true,
        'Sizes - Ordered' => true,
        'Sizes - Price' => true,
        'Total Qty' => true,
        'Total Shipped' => true,
        'Sealer Sample' => true,
        'Shipment Sample' => true,
        'ExFty' => true,
        'Revised ExFty' => true,
        'Customer exFty' => true,
        'Customer Into w/h' => true,
        'Total CMT' => true,
        'Total Sale' => true,
        'Transport Budget' => true,
        'Incoterms' => true,
        'Truck' => true,
        'Collection' => true,
        'Customs ERN' => true,
        'Shipped' => true,
        'Issue' => true,
        'ASOS 3% Discount Price' => false,
        'RT Invoice' => false,
        'Factory Invoice' => false,
        'Customs Invoice' => false,
        'Transporter Invoice' => false,
        'Finance Complete' => false,
    ];

    // Filter dropdown visibility properties
    public $showSeasonDropdown = false;
    public $showCustomerDropdown = false;
    public $showFactoryDropdown = false;
    public $showCooDropdown = false;
    public $showDeptDropdown = false;
    public $showTruckDropdown = false;
    public $showCatDropdown = false;

    // Validation rules
    #[Validate([
        'hideFilters' => ['nullable', 'boolean'],
        'borders' => ['nullable', 'boolean'],
        'view' => ['nullable', 'in:unshipped,financeNC,noRTInvoice,all,cashflow'],
        'fromDate' => ['nullable', 'date', 'before_or_equal:toDate'],
        'toDate' => ['nullable', 'date', 'after_or_equal:fromDate'],
        'department' => ['nullable', 'array', 'exists:departments,id'],
        'customer' => ['nullable', 'array', 'exists:customers,id'],
        'season' => ['nullable', 'array', 'exists:seasons,id'],
        'factory' => ['nullable', 'array', 'exists:suppliers,id'],
        'coo' => ['nullable', 'array', 'exists:countries,id'],
        'category' => ['nullable', 'array'],
        'search' => ['nullable', 'string', 'max:30'],
        'noTruck' => ['nullable', 'boolean'],
        'selected' => ['array'],
        'selected.*' => ['nullable', 'boolean'],
    ])]

    public array $selected = [];

    #[On('updateSelection')]
    public function updateSelection($shipmentLineId, $checked)
    {
        if ($checked) {
                $this->selected[$shipmentLineId] = true;
        } else {
            $this->selected[$shipmentLineId] = false;
        }
    }

    #[On('refresh-columns')]
    public function refresh()
    {
        $this->loadFilterSettings();
        $this->refreshAll();
    }


    public function updated($propertyName)
    {
        // Validate individual field
        $this->validateOnly($propertyName);

        // Check for errors and reset if validation fails
        if ($this->getErrorBag()->has($propertyName)) {
            $this->reset($propertyName);
        } else {
            // Reset pagination only if a relevant property is changed
            if (in_array($propertyName, ['borders', 'view', 'fromDate', 'toDate', 'department', 'customer', 'season', 'factory', 'coo', 'category', 'search', 'truck', 'noTruck', 'hideMabli'])) {
                $this->resetPage();
            }

            // Save filter settings since validation passed
            $this->saveFilterSettings();
            // dd();
            $this->refreshAll();
        }
    }

    protected function filterKey(): string
    {
        return 'shipmentschedulefilters';
    }

    #[On('updateShippedQty')]
    public function updateShippedQty($sizeId, $value)
    {
        \Log::info('updateShippedQty called', ['sizeId' => $sizeId, 'value' => $value]);
        
        try {
            $size = ShipmentLineSizes::find($sizeId);
            if ($size) {
                $size->shipped_qty = $value;
                $size->save();
                \Log::info('Size updated successfully', ['sizeId' => $sizeId, 'newValue' => $value]);
                
                // Force a refresh of the component to show updated values
                $this->dispatch('$refresh');
            } else {
                \Log::error('Size not found', ['sizeId' => $sizeId]);
            }
        } catch (\Exception $e) {
            \Log::error('Error updating size', ['sizeId' => $sizeId, 'error' => $e->getMessage()]);
        }
    }

    public function mount()
    {
        $this->loadFilterSettings();
    }

    public $refreshKey = 0;

    #[On('refreshFullSchedule')]
    public function refreshAll()
    {
        $this->render(); // Update chunks
        $this->refreshKey = now()->timestamp; // Generate a new unique key
    }

    /**
     * Render the component.
     */
    public function render()
    {
        Gate::authorize('shipment:read');

        $this->chunks = $this->getData();

        return view('livewire.production.reports.shipment-schedule');
    }

    public array $chunks;
    public int $page = 1;
    public function hasMorePages()
    {
        return $this->page < count($this->chunks);
    }
    public function loadMore()
    {
        if (!$this->hasMorePages()) {
            return;
        }

        $this->page = $this->page + 1;
    }




    // Computed properties for filters
    #[Computed]
    public function suppliers()
    {
        return Suppliers::allCached()->sortBy('name');
    }

    #[Computed]
    public function factories()
    {
        return Suppliers::whereHas('style_versions')
            ->where('type', 'factory')
            ->with('countries')
            ->get();
    }

    #[Computed]
    public function customers()
    {
        return Customer::whereHas('styles')
            ->get()
            ->sortBy('name');
    }

    #[Computed]
    public function seasons()
    {
        return Seasons::allCached()->sortBy('description');
    }

    #[Computed]
    public function departments()
    {
        return Departments::allCached()->sortBy('name');
    }

    #[Computed]
    public function totalQuoteGBP(){
        // return $this->efficientGetShipment->sum(function ($drop) {
        //     return Conversions::convertCurrency($drop->cust_currency, '£', $drop->total_sale, $this->seasons->where('id', $drop->seasons_id)->first());
        // });
    }
    #[Computed]
    public function totalCmtGBP(){
        // return $this->efficientGetShipment->sum(function ($drop) {
        //     return Conversions::convertCurrency($drop->fact_currency, '£', $drop->total_cmt, $this->seasons->where('id', $drop->seasons_id)->first());
        // });
    }
    #[Computed]
    public function totalCostGBP(){
        // return $this->efficientGetShipment->sum('total_cost');
    }
    #[Computed]
    public function totalMargin(){
        if($this->totalQuoteGBP)
            return (($this->totalQuoteGBP - $this->totalCostGBP) / $this->totalQuoteGBP) * 100;
    }

    // Methods for filter selection
    public function select($filter, $value = 0)
    {
        if (is_string($this->$filter)) {
            $this->$filter = [];
        }
        if ($value == 0) {
            $this->$filter = [];
        } else {
            if (($this->$filter[$value] ?? 0) == 0) {
                $this->$filter[$value] = 1;
            } else {
                unset($this->$filter[$value]);
            }
        }

        $this->updated($filter);
    }

    /**
     * Clear all filters.
     */
    public function clearFilters()
    {
        $this->reset('view', 'fromDate', 'toDate', 'department', 'customer', 'season', 'factory', 'coo', 'category', 'search', 'truck', 'noTruck', 'hideMabli');
    }

    /**
     * Select all drops.
     */
    public function selectAll()
    {
        $this->selected = [];
        $this->dispatch('select-all');
        foreach ($this->chunks as $chunk) {
            foreach($chunk as $sl){
                $this->selected[$sl['id']] = true;
            }
        }
    }

    /**
     * Clear selected drops.
     */
    public function clearSelected()
    {
        $this->selected = [];
        $this->dispatch('clear-selected');
    }

    /**
     * Edit selected drops.
     */
    public function edit()
    {
        // if(Gate::check('ad'))
        //     dd($this->selected);
        $selectedIds = collect($this->selected)
            ->filter() // keeps only truthy values (i.e. true)
            ->keys()   // gets the keys (IDs)
            ->all();   // convert to plain array
        $this->dispatch('edit-ss', selected: json_encode(array_values($selectedIds)));
    }



    /**
     * Get the filter key string.
     */
    public function getFilterKeyString(): string
    {
        return $this->filterKey();
    }

    /**
     * Get the columns version.
     */
    protected function columnsVersion(): float
    {
        return 5;
    }

    /**
     * Get the filters.
     */
    protected function filters(): array
    {
        return ['borders', 'view', 'fromDate', 'toDate', 'department', 'customer', 'season', 'factory', 'coo', 'category', 'search', 'truck', 'noTruck', 'columns', 'hideMabli'];
    }




    /**
     * Update the number of cartons for a drop.
     */
    public function updateFlagIssue($dropId, $value)
    {
        $validatedData = Validator::make(
            ['value' => $value],
            ['value' => 'required|boolean'] // Assuming flag is a binary field (0 or 1)
        )->validate();

        if (Gate::check('order:update')) {
            $drop = ShipmentLine::find($dropId);
            if ($drop) {
                $drop->flag_issue = empty($validatedData['value']) ? 0 : $validatedData['value'];
                $drop->save();
            }
        }
    }




    /**
     * Handle the close modal event.
     */
    #[On('close-modal')]
    public function modalClosed()
    {
        $this->clearSelected();
        $this->render();
    }










    private function getData()
    {
        $query = ShipmentLine::query();

        $query = $this->applyFilters($query)
            ->select([
                'shipment_lines.id',
            ])
            ->join('customer_order_lines', function ($join) {
                $join->on('customer_order_lines.id', '=', 'shipment_lines.customer_order_lines_id')
                    ->where('customer_order_lines.cancelled', 0);
            })
            ->Join('customer_orders', function($join)
                {
                    $join->on('customer_orders.id', '=', 'customer_order_lines.customer_orders_id');
                    $join->on('customer_orders.order_type', '=', DB::raw("'wholesale'"));
                    $join->on('customer_orders.cancelled', '=', DB::raw(0));
                })
            ->join('customers', 'customers.id', '=', 'customer_orders.customers_id')
            ->join('colourways', 'colourways.id', '=', 'customer_order_lines.colourways_id')
            ->join('style_versions', 'style_versions.id', '=', 'colourways.style_versions_id')
            ->join('styles', 'styles.id', '=', 'style_versions.styles_id')
            ->join('seasons', 'seasons.id', '=', 'customer_orders.seasons_id')
            ->join('suppliers as factory', 'factory.id', '=', 'style_versions.factory_id')
            ->leftJoin('shipments', 'shipments.id', '=', 'shipment_lines.shipment_id')
            ->leftJoin('suppliers as transporters', 'transporters.id', '=', 'shipments.transporter_id')


            // ✅ Mark first row using ROW_NUMBER()
            ->selectRaw("
                CASE
                    WHEN ROW_NUMBER() OVER (
                        PARTITION BY shipment_lines.shipment_id
                        ORDER BY
                            -(shipment_lines.complete),
                            -(shipments.truck_first_collection) desc,
                            shipment_lines.shipment_id,
                            -(COALESCE(shipments.truck_first_collection, shipment_lines.exfty)) desc,
                            factory.countries_id,
                            -(factory.name),
                            -(customers.name),
                            -(customer_orders.customer_po),
                            -(styles.id),
                            -(colourways.name),
                            shipment_lines.id
                    ) = 1
                    THEN 1 ELSE 0
                END AS is_first_in_shipment
            ")

            // ✅ Mark last row using COUNT(*) OVER()
            ->selectRaw("
                CASE
                    WHEN ROW_NUMBER() OVER (
                        PARTITION BY shipment_lines.shipment_id
                        ORDER BY
                            -(shipment_lines.complete),
                            -(shipments.truck_first_collection) desc,
                            shipment_lines.shipment_id,
                            -(COALESCE(shipments.truck_first_collection, shipment_lines.exfty)) desc,
                            factory.countries_id,
                            -(factory.name),
                            -(customers.name),
                            -(customer_orders.customer_po),
                            -(styles.id),
                            -(colourways.name),
                            shipment_lines.id
                    ) = COUNT(*) OVER (
                        PARTITION BY shipment_lines.shipment_id
                    )
                    THEN 1 ELSE 0
                END AS is_last_in_shipment
            ")

        ->orderByRaw("
            -(shipment_lines.complete),
            -(shipments.truck_first_collection) desc,
            shipment_lines.shipment_id,
            -(COALESCE(shipments.truck_first_collection, shipment_lines.exfty)) desc,
            factory.countries_id,
            -(factory.name),
            -(customers.name),
            -(customer_orders.customer_po),
            -(styles.id),
            -(colourways.name),
            shipment_lines.id
        ");


        // ✅ Fetch all columns and chunk the results
        $data = $query->get([
            'shipment_lines.id', 'shipment_id', 'is_first_in_shipment', 'is_last_in_shipment'
        ])->chunk(10);

        return $data->toArray();
    }







    private function applyFilters($query){
        //VIEW FILTER
        $query->where('customer_orders.order_type', 'wholesale');

        $query->where('customer_order_lines.cancelled', FALSE);

        //TEMP - REMOVE AFTER USE
        if($this->hideMabli){
            $query->where('customer_orders.customers_id', '!=', 9);
        }


        if($this->view == "unshipped"){
            $query->where('shipment_lines.complete', FALSE);
        }
        elseif($this->view == "financeNC"){
            $query->where('shipment_lines.invoiced', FALSE);
        }
        elseif($this->view == "cashflow"){
            $query->whereNull('shipment_lines.rt_invoice');
        }
        elseif($this->view == "noRTInvoice"){
            $query->where(function($q) {
                $q->whereNull('shipment_lines.rt_invoice')
                ->orWhere('shipment_lines.rt_invoice', '');
            });
            // $query->where('shipment_lines.complete', TRUE); //DO NOT ADD BACK IN - VALERIE
        }
        else{
            $query->where('shipment_lines.exfty', '>=', $this->fromDate);
        }

        //DATE FILTER
        if (!empty($this->toDate)) {
            $query->where('exfty', '<=', $this->toDate);
        }

        //SEASONS FILTER
        if(!empty($this->season)) {
            $query->whereHas('customer_order_lines.customer_orders', function($q){
                $q->whereIn('seasons_id', array_keys($this->season));
            });
        }

        //CUSTOMERS FILTER
        if (!empty($this->customer)) {
            $query->whereIn('customer_orders.customers_id', array_keys($this->customer));
        }

        //FACTORIES FILTER
        if(!empty($this->factory)) {
            $query->whereHas('customer_order_lines.colourways.style_versions', function($q){
                $q->whereIn('factory_id', array_keys($this->factory));
            });
        }

        //COO FILTER
        if(!empty($this->coo)) {
            $query->whereHas('customer_order_lines.colourways.style_versions.factories', function($q){
                $q->whereIn('countries_id', array_keys($this->coo));
            });
        }

        //DEPARTMENT FILTER
        if(!empty($this->department)) {
            $query->whereHas('customer_order_lines.customer_orders', function($q){
                $q->whereIn('departments_id', array_keys($this->department));
            });
        }

        //CATEGORY FILTER
        if(!empty($this->category)) {
            $query->whereHas('customer_order_lines.colourways.style_versions.styles', function($q){
                $q->whereIn('category', array_keys($this->category));
            });
        }

        //TRUCK FILTER
        if ($this->noTruck) {
            $query->whereNull('shipment_lines.shipment_id');
        } elseif (!empty($this->truck)) {
            $query->where('shipments.id', $this->truck);
        }

        //SEARCH FILTER
        if (! empty($this->search)) {
            $query->search($this->search);
        }
        return $query;
    }

}
