<?php

namespace App\Services;

use App\Models\LibraryCategory;
use App\Models\LibraryTrack;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Pagination\LengthAwarePaginator;

class LibraryService
{
    public function getCategories(): Collection
    {
        return LibraryCategory::active()->ordered()->with('tracks')->get();
    }

    public function getCategory(string $slug): ?LibraryCategory
    {
        return LibraryCategory::where('slug', $slug)->active()->with('tracks')->first();
    }

    public function getTracks(array $filters = []): LengthAwarePaginator
    {
        $query = LibraryTrack::active()->with('category');

        if (isset($filters['category_id'])) {
            $query->byCategory($filters['category_id']);
        }

        if (isset($filters['voice_type'])) {
            $query->byVoiceType($filters['voice_type']);
        }

        if (isset($filters['difficulty'])) {
            $query->byDifficulty($filters['difficulty']);
        }

        if (isset($filters['search'])) {
            $query->where(function ($q) use ($filters) {
                $q->where('title', 'like', '%' . $filters['search'] . '%')
                    ->orWhere('description', 'like', '%' . $filters['search'] . '%')
                    ->orWhere('tags', 'like', '%' . $filters['search'] . '%');
            });
        }

        return $query->orderBy('title')->paginate(12);
    }

    public function getTrack(string $slug): ?LibraryTrack
    {
        $track = LibraryTrack::where('slug', $slug)->active()->with('category')->first();
        
        if ($track) {
            $track->incrementPlayCount();
        }

        return $track;
    }

    public function getTrackAudioUrl(LibraryTrack $track): string
    {
        // In a real app, this would generate a signed URL for GCS/S3
        // For demo purposes, we'll return the path directly
        return $track->audio_path;
    }

    public function getPopularTracks(int $limit = 6): Collection
    {
        return LibraryTrack::active()
            ->orderBy('play_count', 'desc')
            ->limit($limit)
            ->with('category')
            ->get();
    }

    public function getTracksByCategory(int $categoryId, int $limit = 8): Collection
    {
        return LibraryTrack::active()
            ->byCategory($categoryId)
            ->limit($limit)
            ->with('category')
            ->get();
    }
}
