<?php

namespace App\Services;

use App\Models\LibraryCategory;
use App\Models\LibraryTrack;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Pagination\LengthAwarePaginator;

class LibraryService
{
    public function getCategories(): Collection
    {
        return LibraryCategory::active()->ordered()->with('tracks')->get();
    }

    public function getTracks(array $filters = []): LengthAwarePaginator
    {
        $query = LibraryTrack::active()->with('category');

        // Apply filters
        if (isset($filters['category_id'])) {
            $query->where('category_id', $filters['category_id']);
        }

        if (isset($filters['difficulty_level'])) {
            $query->byDifficulty($filters['difficulty_level']);
        }

        if (isset($filters['voice_type'])) {
            $query->byVoiceType($filters['voice_type']);
        }

        if (isset($filters['search'])) {
            $search = $filters['search'];
            $query->where(function ($q) use ($search) {
                $q->where('title', 'like', "%{$search}%")
                  ->orWhere('description', 'like', "%{$search}%")
                  ->orWhere('transcript', 'like', "%{$search}%");
            });
        }

        // Apply sorting
        $sortBy = $filters['sort_by'] ?? 'title';
        $sortOrder = $filters['sort_order'] ?? 'asc';
        
        $query->orderBy($sortBy, $sortOrder);

        return $query->paginate($filters['per_page'] ?? 12);
    }

    public function getTrackBySlug(string $slug): ?LibraryTrack
    {
        return LibraryTrack::active()->where('slug', $slug)->with('category')->first();
    }

    public function getCategoryBySlug(string $slug): ?LibraryCategory
    {
        return LibraryCategory::active()->where('slug', $slug)->with('tracks')->first();
    }

    public function incrementPlayCount(LibraryTrack $track): void
    {
        $track->increment('play_count');
    }

    public function searchTracks(string $query, int $limit = 10): Collection
    {
        return LibraryTrack::active()
            ->where(function ($q) use ($query) {
                $q->where('title', 'like', "%{$query}%")
                  ->orWhere('description', 'like', "%{$query}%")
                  ->orWhere('tags', 'like', "%{$query}%");
            })
            ->with('category')
            ->limit($limit)
            ->get();
    }

    public function getPopularTracks(int $limit = 6): Collection
    {
        return LibraryTrack::active()
            ->orderBy('play_count', 'desc')
            ->orderBy('rating', 'desc')
            ->with('category')
            ->limit($limit)
            ->get();
    }

    public function getRecentTracks(int $limit = 6): Collection
    {
        return LibraryTrack::active()
            ->orderBy('created_at', 'desc')
            ->with('category')
            ->limit($limit)
            ->get();
    }
}
