<?php

namespace App\Services;

use App\Models\LibraryCategory;
use App\Models\LibraryTrack;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Pagination\LengthAwarePaginator;

class LibraryService
{
    public function getCategories(): Collection
    {
        return LibraryCategory::active()->ordered()->withCount('tracks')->get();
    }

    public function getCategoryBySlug(string $slug): ?LibraryCategory
    {
        return LibraryCategory::active()->where('slug', $slug)->first();
    }

    public function getTracks(
        ?int $categoryId = null,
        ?string $voiceType = null,
        ?string $difficulty = null,
        ?string $search = null,
        int $perPage = 12
    ): LengthAwarePaginator {
        $query = LibraryTrack::active()->with('category');

        if ($categoryId) {
            $query->byCategory($categoryId);
        }

        if ($voiceType) {
            $query->byVoiceType($voiceType);
        }

        if ($difficulty) {
            $query->byDifficulty($difficulty);
        }

        if ($search) {
            $query->where(function ($q) use ($search) {
                $q->where('title', 'ilike', "%{$search}%")
                  ->orWhere('description', 'ilike', "%{$search}%")
                  ->orWhere('tags', 'ilike', "%{$search}%");
            });
        }

        return $query->orderBy('created_at', 'desc')->paginate($perPage);
    }

    public function getTrackBySlug(string $slug): ?LibraryTrack
    {
        return LibraryTrack::active()->where('slug', $slug)->with('category')->first();
    }

    public function incrementPlayCount(LibraryTrack $track): void
    {
        $track->incrementPlayCount();
    }

    public function getPopularTracks(int $limit = 6): Collection
    {
        return LibraryTrack::active()
            ->with('category')
            ->orderBy('play_count', 'desc')
            ->limit($limit)
            ->get();
    }

    public function getRecentTracks(int $limit = 6): Collection
    {
        return LibraryTrack::active()
            ->with('category')
            ->orderBy('created_at', 'desc')
            ->limit($limit)
            ->get();
    }

    public function searchTracks(string $query, int $perPage = 12): LengthAwarePaginator
    {
        return LibraryTrack::active()
            ->with('category')
            ->where(function ($q) use ($query) {
                $q->where('title', 'ilike', "%{$query}%")
                  ->orWhere('description', 'ilike', "%{$query}%")
                  ->orWhere('tags', 'ilike', "%{$query}%");
            })
            ->orderBy('created_at', 'desc')
            ->paginate($perPage);
    }
}
