<?php

namespace App\Services;

use App\Models\LibraryCategory;
use App\Models\LibraryTrack;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Pagination\LengthAwarePaginator;

class LibraryService
{
    public function getCategories(): Collection
    {
        return LibraryCategory::active()->ordered()->with('tracks')->get();
    }

    public function getCategory(string $slug): ?LibraryCategory
    {
        return LibraryCategory::where('slug', $slug)->active()->with('tracks')->first();
    }

    public function getTracks(
        ?string $categoryId = null,
        ?string $voiceType = null,
        ?string $difficulty = null,
        ?string $search = null,
        int $perPage = 12,
        string $sortBy = 'title',
        string $sortOrder = 'asc'
    ): LengthAwarePaginator {
        $query = LibraryTrack::active()->with('category');

        if ($categoryId) {
            $query->byCategory($categoryId);
        }

        if ($voiceType) {
            $query->byVoiceType($voiceType);
        }

        if ($difficulty) {
            $query->byDifficulty($difficulty);
        }

        if ($search) {
            $query->where(function ($q) use ($search) {
                $q->where('title', 'like', '%' . $search . '%')
                    ->orWhere('description', 'like', '%' . $search . '%')
                    ->orWhere('tags', 'like', '%' . $search . '%');
            });
        }

        // Handle sorting
        $allowedSortFields = ['title', 'rating', 'duration', 'created_at'];
        if (!in_array($sortBy, $allowedSortFields)) {
            $sortBy = 'title';
        }

        $query->orderBy($sortBy, $sortOrder === 'desc' ? 'desc' : 'asc');

        return $query->paginate($perPage);
    }

    public function getTrack(string $slug): ?LibraryTrack
    {
        $track = LibraryTrack::where('slug', $slug)->active()->with('category')->first();
        
        if ($track) {
            $track->incrementPlayCount();
        }

        return $track;
    }

    public function getTrackAudioUrl(LibraryTrack $track): string
    {
        // In a real app, this would generate a signed URL for GCS/S3
        // For demo purposes, we'll return the path directly
        return $track->audio_path;
    }

    public function getPopularTracks(int $limit = 6): Collection
    {
        return LibraryTrack::active()
            ->orderBy('play_count', 'desc')
            ->limit($limit)
            ->with('category')
            ->get();
    }

    public function getTracksByCategory(int $categoryId, int $limit = 8): Collection
    {
        return LibraryTrack::active()
            ->byCategory($categoryId)
            ->limit($limit)
            ->with('category')
            ->get();
    }
}
