<?php

namespace App\Models;

use App\Casts\Boolean;
use App\Casts\Currency;
use App\Casts\TitleCase;
use App\Casts\Percentage;
use App\Models\BaseModel;
use Illuminate\Support\Str;
use \Bkwld\Cloner\Cloneable;
use App\Models\ColourwayYarns;
use Illuminate\Support\Facades\Cache;
use OwenIt\Auditing\Contracts\Auditable;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class ColourwayYarns extends BaseModel implements Auditable
{
    use \OwenIt\Auditing\Auditable;
    use SoftDeletes;
    use Cloneable;

    protected static function booted(): void
    {
        static::saved(function ($colourwayYarn) {
            self::clearPriceCacheForColourway($colourwayYarn);
            self::invalidatePriceResolutions($colourwayYarn);
        });

        static::deleted(function ($colourwayYarn) {
            self::clearPriceCacheForColourway($colourwayYarn);
            self::invalidatePriceResolutions($colourwayYarn);
        });
    }

    protected static function clearPriceCacheForColourway($colourwayYarn)
    {
        // Get the related style version ID and season ID
        $styleVersionId = $colourwayYarn->colourways->style_versions_id;
        $seasonId = $colourwayYarn->colourways->style_versions?->styles?->seasons_id;

        if ($styleVersionId) {
            $cacheTags = [
                "prices",
                "colourway:{$colourwayYarn->colourways_id}",
                "styleversion:{$styleVersionId}"
            ];
            
            if ($seasonId) {
                $cacheTags[] = "season:{$seasonId}";
            }
            
            // Clear price cache using consistent tags
            Cache::tags($cacheTags)->flush();

            // Also clear yarn cost cache with consistent tags
            Cache::tags(['yarn_costs', "style_version:{$styleVersionId}"])->flush();
        } else {
            // Fallback to just colourway cache if no style version
            Cache::tags(["prices", "colourway:{$colourwayYarn->colourways_id}"])->flush();
        }
    }

    protected $casts = [
        'price' => Currency::class,
        'percentage' => Percentage::class,
        'CustRef' => TitleCase::class,
        'created_at' => 'datetime:Y-m-d',
        'updated_at' => 'datetime:Y-m-d',
        'deleted_at' => 'datetime:Y-m-d',
        'not_full_factored' => Boolean::class,
    ];

	protected $fillable = [
        'colourways_id',
        'cust_ref',
        'percentage',
        'price',
        'not_full_factored',
        'created_at',
        'updated_at',
        'yarn_colours_id',
        'deleted_at',
    ];

    // protected function CustRef(): Attribute
    // {
    //     return Attribute::make(
    //         get: fn ($value) => Str::title($value),
    //     );
    // }

    // protected function price(): Attribute
    // {
    //     return Attribute::make(
    //         get: fn ($value) => number_format($value, 2, '.', ''),
    //         set: fn ($value) => empty($value) ? 0 : (preg_replace("/[^0-9.]/", "", $value) + 0),
    //     );
    // }
    // protected function percentage(): Attribute
    // {
    //     return Attribute::make(
    //         get: fn ($value) => number_format($value, 0, '.', ''),
    //         set: fn ($value) => empty($value) ? 0 : (preg_replace("/[^0-9.]/", "", $value) + 0),
    //     );
    // }

    public function colourways()
    {
        return $this->belongsTo(Colourways::class);
    }
    public function yarn_colours()
    {
        return $this->belongsTo(YarnColours::class);
    }

    protected static function invalidatePriceResolutions($colourwayYarn)
    {
        try {
            $service = app(\App\Services\PriceResolutionService::class);
            // Invalidate by colourway since yarn changes affect the colourway's price calculations
            $service->invalidateByColourway($colourwayYarn->colourways_id);
        } catch (\Exception $e) {
            \Log::error('Failed to invalidate price resolutions for colourway yarn', [
                'colourway_yarn_id' => $colourwayYarn->id,
                'colourway_id' => $colourwayYarn->colourways_id,
                'error' => $e->getMessage()
            ]);
        }
    }
}
