<?php

namespace App\Services;

use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Carbon\Carbon;

class MediaService
{
    /**
     * Store an uploaded file and return the file path
     */
    public function storeFile($file, string $directory = 'uploads'): string
    {
        $filename = $this->generateUniqueFilename($file);
        $path = $file->storeAs($directory, $filename, $this->getDefaultDisk());
        
        return $path;
    }

    /**
     * Generate a signed URL for a file with TTL
     */
    public function generateSignedUrl(string $filePath, int $ttlMinutes = 60): ?string
    {
        if (empty($filePath)) {
            return null;
        }

        try {
            $disk = Storage::disk($this->getDefaultDisk());
            
            if (!$disk->exists($filePath)) {
                return null;
            }

            // For S3, use temporary URLs
            if ($this->getDefaultDisk() === 's3') {
                return $disk->temporaryUrl($filePath, Carbon::now()->addMinutes($ttlMinutes));
            }

            // For local storage, create a temporary signed URL
            return $this->createLocalSignedUrl($filePath, $ttlMinutes);
        } catch (\Exception $e) {
            \Log::error('Failed to generate signed URL for: ' . $filePath, [
                'error' => $e->getMessage(),
                'file_path' => $filePath
            ]);
            return null;
        }
    }

    /**
     * Delete a file from storage
     */
    public function deleteFile(string $filePath): bool
    {
        if (empty($filePath)) {
            return false;
        }

        try {
            return Storage::disk($this->getDefaultDisk())->delete($filePath);
        } catch (\Exception $e) {
            \Log::error('Failed to delete file: ' . $filePath, [
                'error' => $e->getMessage(),
                'file_path' => $filePath
            ]);
            return false;
        }
    }

    /**
     * Check if a file exists
     */
    public function fileExists(string $filePath): bool
    {
        if (empty($filePath)) {
            return false;
        }

        return Storage::disk($this->getDefaultDisk())->exists($filePath);
    }

    /**
     * Get file size in bytes
     */
    public function getFileSize(string $filePath): ?int
    {
        if (empty($filePath)) {
            return null;
        }

        try {
            return Storage::disk($this->getDefaultDisk())->size($filePath);
        } catch (\Exception $e) {
            return null;
        }
    }

    /**
     * Get file duration for audio files (placeholder for actual implementation)
     */
    public function getAudioDuration(string $filePath): ?int
    {
        // This would typically use a library like getID3 or ffprobe
        // For now, return null as placeholder
        return null;
    }

    /**
     * Generate a unique filename for uploaded files
     */
    private function generateUniqueFilename($file): string
    {
        $extension = $file->getClientOriginalExtension();
        $basename = Str::slug(pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME));
        $timestamp = now()->timestamp;
        $random = Str::random(8);
        
        return "{$basename}-{$timestamp}-{$random}.{$extension}";
    }

    /**
     * Get the default storage disk
     */
    private function getDefaultDisk(): string
    {
        return config('filesystems.default');
    }

    /**
     * Create a local signed URL (for development/testing)
     */
    private function createLocalSignedUrl(string $filePath, int $ttlMinutes): string
    {
        $token = Str::random(32);
        $expires = now()->addMinutes($ttlMinutes)->timestamp;
        
        // Store the token temporarily (in a real app, you might use Redis or cache)
        cache()->put("signed_url_{$token}", [
            'file_path' => $filePath,
            'expires' => $expires
        ], $ttlMinutes * 60);
        
        return route('media.signed', ['token' => $token]);
    }

    /**
     * Validate a signed URL token
     */
    public function validateSignedUrlToken(string $token): ?array
    {
        $data = cache()->get("signed_url_{$token}");
        
        if (!$data || $data['expires'] < now()->timestamp) {
            return null;
        }
        
        return $data;
    }

    /**
     * Get the public URL for a file (if using public disk)
     */
    public function getPublicUrl(string $filePath): ?string
    {
        if (empty($filePath)) {
            return null;
        }

        try {
            $disk = Storage::disk($this->getDefaultDisk());
            
            if ($this->getDefaultDisk() === 'public') {
                return $disk->url($filePath);
            }
            
            return null;
        } catch (\Exception $e) {
            return null;
        }
    }
}
