<?php

namespace App\Jobs;

use App\Services\EMLProcessingService;
use Illuminate\Bus\Batchable;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;

class ProcessEMLReceipt implements ShouldQueue
{
    use Batchable, Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public int $tries = 3;
    public int $timeout = 60;
    public int $maxExceptions = 2;

    public function __construct(
        public string $emlPath,
        public int $userId
    ) {}

    public function backoff(): array
    {
        return [15, 60, 300];
    }

    public function retryUntil(): \DateTimeInterface
    {
        return now()->addHours(6);
    }

    public function handle(EMLProcessingService $emlService): void
    {
        try {
            $receiptData = $emlService->processEMLFile($this->emlPath);
            
            if ($receiptData) {
                $receipt = $emlService->createReceiptFromEML($this->emlPath, $receiptData, $this->userId);
                \Log::info('EML receipt processed successfully', [
                    'receipt_id' => $receipt->id,
                    'eml_path' => $this->emlPath
                ]);
            }
        } catch (\Throwable $e) {
            \Log::warning('ProcessEMLReceipt: transient failure, will retry', [
                'eml_path' => $this->emlPath,
                'error' => $e->getMessage(),
                'attempt' => method_exists($this, 'attempts') ? $this->attempts() : null,
            ]);
            $schedule = (array) $this->backoff();
            $index = max(0, min((int)($this->attempts() - 1), count($schedule) - 1));
            $this->release((int) $schedule[$index]);
        }
    }
}
