<?php

namespace App\Livewire;

use App\Models\Consent;
use App\Models\User;
use Illuminate\Support\Facades\Auth;
use Livewire\Component;
use Livewire\Attributes\On;

class ConsentManagement extends Component
{
    public User $user;
    public $consents = [];
    public $showRevokeModal = false;
    public $consentToRevoke = null;
    public $revokeReason = '';

    public function mount()
    {
        $this->user = Auth::user();
        $this->loadConsents();
    }

    public function loadConsents()
    {
        $this->consents = $this->user->consents()
            ->orderBy('type')
            ->orderBy('created_at', 'desc')
            ->get()
            ->groupBy('type');
    }

    public function acceptConsent($consentId)
    {
        $consent = Consent::findOrFail($consentId);
        
        if ($consent->user_id !== $this->user->id) {
            $this->addError('consent', 'You can only manage your own consents.');
            return;
        }

        $consent->accept();
        $this->loadConsents();
        
        $this->dispatch('consent-updated', [
            'message' => 'Consent accepted successfully.',
            'type' => 'success'
        ]);
    }

    public function revokeConsent($consentId)
    {
        $consent = Consent::findOrFail($consentId);
        
        if ($consent->user_id !== $this->user->id) {
            $this->addError('consent', 'You can only manage your own consents.');
            return;
        }

        $consent->revoke();
        $this->loadConsents();
        
        $this->dispatch('consent-updated', [
            'message' => 'Consent revoked successfully.',
            'type' => 'success'
        ]);
    }

    public function openRevokeModal($consentId)
    {
        $this->consentToRevoke = Consent::findOrFail($consentId);
        $this->showRevokeModal = true;
        $this->revokeReason = '';
    }

    public function closeRevokeModal()
    {
        $this->showRevokeModal = false;
        $this->consentToRevoke = null;
        $this->revokeReason = '';
    }

    public function confirmRevoke()
    {
        if ($this->consentToRevoke) {
            $this->revokeConsent($this->consentToRevoke->id);
            $this->closeRevokeModal();
        }
    }

    public function getConsentStatus($consent)
    {
        if (!$consent->is_accepted) {
            return 'Not Accepted';
        }
        
        if ($consent->isExpired()) {
            return 'Expired';
        }
        
        return 'Active';
    }

    public function getConsentStatusColor($consent)
    {
        if (!$consent->is_accepted) {
            return 'text-red-600 bg-red-100';
        }
        
        if ($consent->isExpired()) {
            return 'text-orange-600 bg-orange-100';
        }
        
        return 'text-green-600 bg-green-100';
    }

    public function render()
    {
        return view('livewire.consent-management');
    }
}
