<?php

namespace App\Jobs;

use App\Models\ReceiptGroup;
use App\Services\Matching\GroupedReceiptMatchingService;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;

class ProcessReceiptGroup implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public int $tries = 3;
    public int $timeout = 60;

    public function __construct(public int $receiptGroupId) {}

    public function backoff(): array
    {
        return [15, 60, 300];
    }

    public function retryUntil(): \DateTimeInterface
    {
        return now()->addHours(6);
    }

    public function handle(GroupedReceiptMatchingService $matchingService): void
    {
        $group = ReceiptGroup::find($this->receiptGroupId);
        
        if (!$group) {
            \Log::warning('ReceiptGroup not found for processing', ['group_id' => $this->receiptGroupId]);
            return;
        }

        if ($group->hasMatches()) {
            \Log::info('ReceiptGroup already has matches, skipping', ['group_id' => $this->receiptGroupId]);
            return;
        }

        try {
            $matchingService->attemptMatchForGroup($group);
        } catch (\Throwable $e) {
            \Log::error('Failed to process receipt group', [
                'group_id' => $this->receiptGroupId,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            throw $e;
        }
    }
}