#!/usr/bin/env node

const { Client, Pool } = require('pg');
const bcrypt = require('bcrypt');
const fs = require('fs');

require('dotenv').config();

// Connection for creating database (connect to postgres database)
const adminConfig = {
  user: process.env.DB_USER || 'postgres',
  password: process.env.DB_PASSWORD,
  host: process.env.DB_HOST || 'localhost',
  port: parseInt(process.env.DB_PORT || '5432', 10),
  database: 'postgres' // Connect to default postgres database
};

// Connection for app database
const dbName = process.env.DB_NAME || 'photowebsite';
const dbConfig = {
  user: process.env.DB_USER || 'postgres',
  password: process.env.DB_PASSWORD,
  host: process.env.DB_HOST || 'localhost',
  port: parseInt(process.env.DB_PORT || '5432', 10),
  database: dbName
};

async function createDatabase() {
  console.log('Step 1: Ensuring database exists...\n');
  
  const client = new Client(adminConfig);
  
  try {
    await client.connect();
    
    // Check if database exists
    const result = await client.query(
      "SELECT 1 FROM pg_database WHERE datname = $1",
      [dbName]
    );
    
    if (result.rows.length === 0) {
      console.log(`Creating database "${dbName}"...`);
      await client.query(`CREATE DATABASE ${dbName}`);
      console.log('✓ Database created successfully\n');
    } else {
      console.log(`✓ Database "${dbName}" already exists\n`);
    }
  } catch (error) {
    console.error('Error creating database:', error.message);
    throw error;
  } finally {
    await client.end();
  }
}

async function runMigrations() {
  console.log('Step 2: Running migrations...\n');
  
  const pool = new Pool(dbConfig);
  
  try {
    // Run the first migration (001)
    const migration1Path = './migrations/001-multi-tenant-schema.sql';
    if (fs.existsSync(migration1Path)) {
      console.log('Running 001-multi-tenant-schema.sql...');
      const sql1 = fs.readFileSync(migration1Path, 'utf8');
      await pool.query(sql1);
      console.log('✓ Multi-tenant schema migration completed\n');
    }
    
    // Run the superuser migration (002)
    const migration2Path = './migrations/002-add-superuser-support.sql';
    if (fs.existsSync(migration2Path)) {
      console.log('Running 002-add-superuser-support.sql...');
      const sql2 = fs.readFileSync(migration2Path, 'utf8');
      await pool.query(sql2);
      console.log('✓ Superuser support migration completed\n');
    }
    
  } catch (error) {
    if (error.message.includes('already exists')) {
      console.log('✓ Tables already exist, continuing...\n');
    } else {
      console.error('Migration error:', error.message);
      throw error;
    }
  } finally {
    await pool.end();
  }
}

async function createSuperuser() {
  console.log('Step 3: Creating superuser account...\n');
  
  const pool = new Pool(dbConfig);
  
  // Default credentials
  const username = 'admin';
  const email = 'admin@vybephoto.com';
  const password = 'admin123456';
  
  console.log('Using default superuser credentials:');
  console.log('  Username:', username);
  console.log('  Email:', email);
  console.log('  Password:', password);
  console.log('  ⚠️  CHANGE THE PASSWORD AFTER FIRST LOGIN!\n');
  
  try {
    console.log('Generating password hash...');
    const passwordHash = await bcrypt.hash(password, 10);
    
    console.log('Creating/updating superuser account...');
    
    const result = await pool.query(
      `INSERT INTO users (
        username, 
        email, 
        password_hash, 
        full_name,
        subscription_tier, 
        subscription_status,
        is_superuser,
        max_photos,
        max_storage_bytes
      ) VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9)
      ON CONFLICT (username) 
      DO UPDATE SET
        is_superuser = TRUE,
        subscription_status = 'active',
        subscription_tier = 'business',
        password_hash = EXCLUDED.password_hash
      RETURNING id, username, email, is_superuser`,
      [
        username.toLowerCase(),
        email.toLowerCase(),
        passwordHash,
        'Superuser Administrator',
        'business',
        'active',
        true,
        999999,
        999999999999
      ]
    );
    
    console.log('\n===================================');
    console.log('✓ Superuser ready!');
    console.log('===================================');
    console.log('User ID:', result.rows[0].id);
    console.log('Username:', result.rows[0].username);
    console.log('Email:', result.rows[0].email);
    console.log('Superuser:', result.rows[0].is_superuser);
    
  } catch (error) {
    if (error.code === '23505' && error.constraint === 'users_email_key') {
      // Email already exists, try updating by email
      console.log('\nEmail already in use, updating that user to superuser...\n');
      
      const result = await pool.query(
        `UPDATE users 
         SET is_superuser = TRUE,
             subscription_status = 'active',
             subscription_tier = 'business'
         WHERE email = $1
         RETURNING id, username, email, is_superuser`,
        [email.toLowerCase()]
      );
      
      if (result.rows.length > 0) {
        console.log('✓ User updated to superuser');
        console.log('Username:', result.rows[0].username);
        console.log('Email:', result.rows[0].email);
      }
    } else {
      console.error('Error creating superuser:', error.message);
      throw error;
    }
  } finally {
    await pool.end();
  }
}

async function main() {
  console.log('===================================');
  console.log('Vybe Photo - Complete Setup');
  console.log('===================================\n');
  
  console.log('Configuration:');
  console.log('  Host:', dbConfig.host);
  console.log('  Port:', dbConfig.port);
  console.log('  Database:', dbConfig.database);
  console.log('  User:', dbConfig.user);
  console.log('');
  
  try {
    await createDatabase();
    await runMigrations();
    await createSuperuser();
    
    console.log('\n===================================');
    console.log('✅ Setup Complete!');
    console.log('===================================\n');
    console.log('Login Credentials:');
    console.log('  URL: http://app.localhost:3000/login');
    console.log('  Username: admin');
    console.log('  Password: admin123456');
    console.log('\nSuperuser Portal:');
    console.log('  http://app.localhost:3000/superuser');
    console.log('\n⚠️  IMPORTANT: Change your password after first login!');
    console.log('\nNext step: Start the server with:');
    console.log('  node src/server-multi-tenant.js\n');
    
  } catch (error) {
    console.error('\n✗ Setup failed:', error.message);
    process.exit(1);
  }
}

main();

