<?php

namespace App\Livewire;

use Livewire\Component;
use Livewire\WithPagination;
use App\Models\Season;
use App\Models\Item;
use App\Models\Category;

class ShopSeason extends Component
{
    use WithPagination;

    public $seasonSlug;
    public $season;
    public $selectedCategory = null;
    public $sortBy = 'latest';

    protected $queryString = ['selectedCategory', 'sortBy'];

    public function mount($seasonSlug)
    {
        $this->seasonSlug = $seasonSlug;
        $this->season = Season::where('slug', $seasonSlug)
            ->firstOrFail();
    }

    public function updatedSelectedCategory()
    {
        $this->resetPage();
    }

    public function updatedSortBy()
    {
        $this->resetPage();
    }

    public function render()
    {
        $query = Item::where('seasons_id', $this->season->id)
            ->with(['season', 'category', 'colourways' => function($q) {
                $q->orderBy('sort_order');
            }]);

        if ($this->selectedCategory) {
            $query->whereHas('category', function($q) {
                $q->where('slug', $this->selectedCategory);
            });
        }

        switch ($this->sortBy) {
            case 'price_asc':
                $query->orderBy('price', 'asc');
                break;
            case 'price_desc':
                $query->orderBy('price', 'desc');
                break;
            case 'name':
                $query->orderBy('name', 'asc');
                break;
            default:
                $query->latest();
        }

        $products = $query->paginate(12);

        $categories = Category::whereHas('items', function($q) {
                $q->where('seasons_id', $this->season->id);
            })
            ->orderBy('sort_order')
            ->get();

        return view('livewire.shop-season', [
            'products' => $products,
            'categories' => $categories,
        ])->layout('layouts.shop');
    }
}

