<?php

namespace App\Services\Matching;

use App\Models\ClassificationRule;
use App\Models\Receipt;
use App\Models\ReceiptLine;
use App\Models\StatementTransaction;
use App\Services\AI\VertexClient;
use App\Services\AI\ExpenseClassificationAgent;

class ClassificationService
{
    public function __construct(
        private VertexClient $vertex,
        private ExpenseClassificationAgent $expenseAgent
    ) {}

    public function classifyReceipt(Receipt $receipt): void
    {
        // LLM classification with fallback to rules
        $model = config('vertex.models.classify', 'gemini-2.5-flash-lite');
        $system = 'You help categorize expenses for UK business accounting. Given a receipt merchant and optional line items, decide department and account code, and whether potentially personal. Output JSON with keys department, account, is_personal (boolean). Use provided allowed values if present.';
        $prompt = json_encode([
            'merchant' => $receipt->merchant_name,
            'total_amount' => $receipt->total_amount,
            'lines' => $receipt->lines()->get(['description','line_total'])->toArray(),
            'allowed_departments' => ClassificationService::allowedDepartments(),
            'allowed_accounts' => ClassificationService::allowedAccounts(),
        ]);
        $resp = $this->vertex->generate($model, $prompt, $system, ['responseMimeType' => 'application/json']);
        $json = $resp['json'];
        if (is_array($json)) {
            if (!empty($json['department'])) {
                $receipt->department_id = ClassificationService::mapDepartmentNameToId($json['department']) ?? $receipt->department_id;
            }
            if (!empty($json['account'])) {
                $receipt->account_id = ClassificationService::mapAccountNameToId($json['account']) ?? $receipt->account_id;
            }
            if (isset($json['is_personal'])) {
                $receipt->is_personal = (bool)$json['is_personal'];
            }
            $receipt->save();
        }

        // Simple rules-based baseline using stored patterns
        if ($receipt->merchant_name) {
            // First try user-specific rules, then global rules
            $rule = ClassificationRule::where('scope', 'merchant')
                ->where('pattern', 'like', '%'.addcslashes($receipt->merchant_name, '%_').'%')
                ->where(function($query) use ($receipt) {
                    $query->where('user_id', $receipt->user_id)
                          ->orWhereNull('user_id');
                })
                ->orderByDesc('user_id') // User-specific rules first
                ->orderByDesc('weight')
                ->orderByDesc('usage_count') // Most used rules first
                ->first();
            if ($rule) {
                $receipt->department_id = $rule->department_id ?? $receipt->department_id;
                $receipt->account_id = $rule->account_id ?? $receipt->account_id;
                $receipt->is_personal = $rule->is_personal || $receipt->is_personal;
                $receipt->save();
                
                // Record usage of this rule
                $rule->recordUsage();
            }
        }
    }

    public function classifyReceiptLine(ReceiptLine $line): void
    {
        // LLM suggestion (optional)
        $model = config('vertex.models.classify', 'gemini-1.5-flash');
        $system = 'Suggest department and account for this receipt line. Output JSON with department and account.';
        $prompt = json_encode(['description' => $line->description, 'allowed_departments' => self::allowedDepartments(), 'allowed_accounts' => self::allowedAccounts()]);
        $resp = $this->vertex->generate($model, $prompt, $system, ['responseMimeType' => 'application/json']);
        $json = $resp['json'];
        if (is_array($json)) {
            $dep = self::mapDepartmentNameToId($json['department'] ?? null);
            $acc = self::mapAccountNameToId($json['account'] ?? null);
            if ($dep) $line->department_id = $dep;
            if ($acc) $line->account_id = $acc;
            $line->save();
        }

        $rule = ClassificationRule::where('scope', 'line_description')
            ->where('pattern', 'like', '%'.addcslashes($line->description, '%_').'%')
            ->where(function($query) use ($line) {
                $query->where('user_id', $line->receipt?->user_id)
                      ->orWhereNull('user_id');
            })
            ->orderByDesc('user_id') // User-specific rules first
            ->orderByDesc('weight')
            ->orderByDesc('usage_count') // Most used rules first
            ->first();
        if ($rule) {
            $line->department_id = $rule->department_id ?? $line->department_id;
            $line->account_id = $rule->account_id ?? $line->account_id;
            $line->is_personal = $rule->is_personal || $line->is_personal;
            $line->save();
            
            // Record usage of this rule
            $rule->recordUsage();
        }
    }

    /**
     * Classify a statement transaction using the new expense agent
     */
    public function classifyStatementTransaction(StatementTransaction $transaction, ?Receipt $receipt = null): array
    {
        return $this->expenseAgent->classifyTransaction($transaction, $receipt);
    }

    /**
     * Re-classify a statement transaction when receipt is matched
     */
    public function reclassifyStatementTransaction(StatementTransaction $transaction, Receipt $receipt): array
    {
        return $this->expenseAgent->reclassifyWithReceipt($transaction, $receipt);
    }

    public static function allowedDepartments(): array
    {
        return \App\Models\Department::pluck('name')->all();
    }

    public static function allowedAccounts(): array
    {
        return \App\Models\Account::pluck('name')->all();
    }

    public static function mapDepartmentNameToId(?string $name): ?int
    {
        if (!$name) return null;
        $m = \App\Models\Department::where('name', $name)->first();
        return $m?->id;
    }

    public static function mapAccountNameToId(?string $name): ?int
    {
        if (!$name) return null;
        $m = \App\Models\Account::where('name', $name)->first();
        return $m?->id;
    }
}


