<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Receipt;
use App\Models\ReceiptItem;
use App\Services\ReceiptStorageService;
use App\Services\MatchingService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\Rule;

class ReceiptIngestController extends Controller
{
    public function store(Request $request)
    {
        $data = $request->validate([
            'receipt_date' => ['required', 'date'],
            'total_amount' => ['required', 'numeric'],
            'currency' => ['nullable', 'string', 'max:8'],
            'vendor_name' => ['nullable', 'string', 'max:255'],
            'is_personal' => ['nullable', 'boolean'],
            'department' => ['nullable', 'string', 'max:255'],
            'account_code' => ['nullable', 'string', 'max:255'],
            'items' => ['nullable', 'array'],
            'items.*.description' => ['required_with:items', 'string'],
            'items.*.quantity' => ['nullable', 'numeric'],
            'items.*.unit_price' => ['nullable', 'numeric'],
            'items.*.line_total' => ['nullable', 'numeric'],
            'image' => ['required', 'file', 'mimes:jpg,jpeg,png,pdf'],
        ]);

        return DB::transaction(function () use ($request, $data) {
            $storageService = app(ReceiptStorageService::class);
            $path = $storageService->store($request->file('image'));

            $receipt = Receipt::create([
                'user_id' => $request->user()->id,
                'receipt_date' => $data['receipt_date'],
                'total_amount' => $data['total_amount'],
                'currency' => $data['currency'] ?? 'GBP',
                'vendor_name' => $data['vendor_name'] ?? null,
                'is_personal' => (bool)($data['is_personal'] ?? false),
                'storage_path' => $path,
                'department' => $data['department'] ?? null,
                'account_code' => $data['account_code'] ?? null,
                'status' => 'uploaded',
            ]);

            foreach (($data['items'] ?? []) as $item) {
                $receipt->items()->create([
                    'description' => $item['description'],
                    'quantity' => $item['quantity'] ?? 1,
                    'unit_price' => $item['unit_price'] ?? 0,
                    'line_total' => $item['line_total'] ?? 0,
                ]);
            }

            app(MatchingService::class)->attemptMatchForReceipt($receipt);

            return response()->json([
                'id' => $receipt->id,
                'status' => $receipt->status,
            ], 201);
        });
    }
}


