<?php

namespace App\Livewire\Manage;

use Livewire\Component;
use Livewire\WithFileUploads;
use Illuminate\Validation\Rules\File;
use App\Services\StatementImportService;
use Illuminate\Support\Facades\Storage;

class ImportStatement extends Component
{
    use WithFileUploads;

    public $month;
    public $year;
    public $file;
    // Simplified per requirement: only month, year, and single file

    protected function rules(): array
    {
        return [
            'month' => ['required','integer','min:1','max:12'],
            'year' => ['required','integer','min:2000','max:2100'],
            'file' => ['required', File::types(['xlsx','xls','csv'])->max('20mb')],
        ];
    }

    public function mount(): void
    {
        $this->month = $this->month ?: (int) now()->month;
        $this->year = $this->year ?: (int) now()->year;
    }

    public function submit(StatementImportService $importer)
    {
        $this->validate();

        $periodStart = sprintf('%04d-%02d-01', (int)$this->year, (int)$this->month);
        $periodEnd = date('Y-m-t', strtotime($periodStart));

        $disk = config('filesystems.default');
        $path = $this->file->store('statements', $disk);
        $absPath = Storage::disk($disk)->path($path);

        // For simplified flow, we derive placeholders and let review handle details if needed
        $result = $importer->import(
            $absPath,
            auth()->user()->name ?? 'Unknown',
            '****',
            $periodStart,
            $periodEnd,
            $this->file->getClientOriginalName(),
            $path,
        );

        return redirect()->route('manage', ['tab' => 'review', 'statementId' => $result['statement_id']]);
    }

    public function render()
    {
        return view('livewire.manage.import-statement');
    }
}


