<?php

namespace App\Livewire\Manage;

use Livewire\Component;
use Livewire\WithFileUploads;
use Illuminate\Validation\Rules\File;
use App\Services\StatementImportService;
use Illuminate\Support\Facades\Storage;

class ImportStatement extends Component
{
    use WithFileUploads;

    public $month;
    public $year;
    public $file;
    public $cardholder_name;
    public $account_number_masked;

    protected function rules(): array
    {
        return [
            'month' => ['required','integer','min:1','max:12'],
            'year' => ['required','integer','min:2000','max:2100'],
            'cardholder_name' => ['required','string','max:255'],
            'account_number_masked' => ['required','string','max:32'],
            'file' => ['required', File::types(['xlsx','xls','csv'])->max('20mb')],
        ];
    }

    public function submit(StatementImportService $importer)
    {
        $this->validate();

        $periodStart = sprintf('%04d-%02d-01', (int)$this->year, (int)$this->month);
        $periodEnd = date('Y-m-t', strtotime($periodStart));

        $disk = config('filesystems.default');
        $path = $this->file->store('statements', $disk);
        $absPath = Storage::disk($disk)->path($path);

        $result = $importer->import(
            $absPath,
            $this->cardholder_name,
            $this->account_number_masked,
            $periodStart,
            $periodEnd,
            $this->file->getClientOriginalName(),
            $path,
        );

        session()->flash('status', 'Imported statement #'.$result['statement_id'].' with '.$result['transactions_imported'].' transactions.');
        $this->reset(['file']);
    }

    public function render()
    {
        return view('livewire.manage.import-statement');
    }
}


