//
//  ContentView.swift
//  Jens Expenses
//
//  Created by Neil Widdowson on 01/10/2025.
//

import SwiftUI
import UIKit
import PhotosUI

struct ContentView: View {
    @StateObject private var viewModel = ReceiptCaptureViewModel()
    @State private var isShowingCamera = false
    @State private var isShowingPhotoPicker = false

    var body: some View {
        NavigationStack {
            VStack(spacing: 16) {
                if viewModel.images.isEmpty {
                    ContentPlaceholder()
                        .frame(maxHeight: 220)
                } else {
                    ScrollView {
                        LazyVGrid(columns: Array(repeating: GridItem(.flexible(), spacing: 12), count: 3), spacing: 12) {
                            ForEach(Array(viewModel.images.enumerated()), id: \.offset) { index, image in
                                ZStack(alignment: .topTrailing) {
                                    Image(uiImage: image)
                                        .resizable()
                                        .scaledToFill()
                                        .frame(height: 110)
                                        .clipped()
                                        .clipShape(RoundedRectangle(cornerRadius: 10))
                                        .overlay(
                                            RoundedRectangle(cornerRadius: 10)
                                                .stroke(Color.secondary.opacity(0.15), lineWidth: 1)
                                        )
                                    Button {
                                        viewModel.removeImage(at: index)
                                    } label: {
                                        Image(systemName: "xmark.circle.fill")
                                            .foregroundStyle(.white, .black.opacity(0.5))
                                    }
                                    .padding(6)
                                }
                            }
                        }
                        .padding(.top, 4)
                    }
                }

                if let error = viewModel.errorMessage {
                    Text(error)
                        .foregroundStyle(.red)
                        .font(.footnote)
                        .frame(maxWidth: .infinity, alignment: .leading)
                }

                if let info = viewModel.infoMessage {
                    Text(info)
                        .foregroundStyle(.green)
                        .font(.footnote)
                        .frame(maxWidth: .infinity, alignment: .leading)
                }

                Spacer(minLength: 0)

                VStack(spacing: 10) {
                    HStack {
                        Button {
                            isShowingCamera = true
                        } label: {
                            Label("Camera", systemImage: "camera")
                                .frame(maxWidth: .infinity)
                        }
                        .buttonStyle(.borderedProminent)

                        Button {
                            isShowingPhotoPicker = true
                        } label: {
                            Label("Photos", systemImage: "photo.on.rectangle")
                                .frame(maxWidth: .infinity)
                        }
                        .buttonStyle(.bordered)
                    }

                    HStack {
                        Button(role: .destructive) {
                            viewModel.clearAll()
                        } label: {
                            Label("Clear", systemImage: "trash")
                                .frame(maxWidth: .infinity)
                        }
                        .buttonStyle(.bordered)
                        .disabled(viewModel.images.isEmpty)

                        ShareImagesButton(images: viewModel.images) {
                            viewModel.infoMessage = "Exported."
                        }
                        .frame(maxWidth: .infinity)
                        .disabled(viewModel.images.isEmpty)
                    }
                }
            }
            .padding()
            .navigationTitle("Receipts")
            .sheet(isPresented: $isShowingCamera) {
                ImagePicker(sourceType: .camera) { image in
                    isShowingCamera = false
                    if let image { viewModel.addImage(image) }
                }
                .ignoresSafeArea()
            }
            .sheet(isPresented: $isShowingPhotoPicker) {
                PhotoPicker { images in
                    isShowingPhotoPicker = false
                    viewModel.addImages(images)
                }
            }
        }
    }
}

private struct ContentPlaceholder: View {
    var body: some View {
        VStack(spacing: 8) {
            Image(systemName: "doc.text.viewfinder")
                .font(.system(size: 42))
                .foregroundStyle(.secondary)
            Text("Capture a receipt to begin")
                .foregroundStyle(.secondary)
        }
        .frame(maxWidth: .infinity)
        .padding()
        .background(
            RoundedRectangle(cornerRadius: 12)
                .fill(Color.secondary.opacity(0.08))
        )
    }
}

private struct ShareImagesButton: View {
    let images: [UIImage]
    var onComplete: () -> Void

    @State private var isPresenting = false

    var body: some View {
        Button {
            isPresenting = true
        } label: {
            Label("Export", systemImage: "square.and.arrow.up")
        }
        .buttonStyle(.borderedProminent)
        .sheet(isPresented: $isPresenting) {
            ShareSheet(activityItems: images, completion: {
                onComplete()
                isPresenting = false
            })
        }
    }
}

private struct ShareSheet: UIViewControllerRepresentable {
    let activityItems: [Any]
    var completion: () -> Void

    func makeUIViewController(context: Context) -> UIActivityViewController {
        let controller = UIActivityViewController(activityItems: activityItems, applicationActivities: nil)
        controller.completionWithItemsHandler = { _, _, _, _ in
            completion()
        }
        return controller
    }

    func updateUIViewController(_ uiViewController: UIActivityViewController, context: Context) {}
}

#Preview {
    ContentView()
}

