<?php

namespace App\Models;

use App\Casts\Upper;
use App\Models\BaseModel;
use OwenIt\Auditing\Contracts\Auditable;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Factories\HasFactory;

/**
 * Gauge model
 * 
 * Represents knitting gauge specifications used in the system.
 */
class Gauge extends BaseModel implements Auditable
{
    use \OwenIt\Auditing\Auditable;
    use SoftDeletes;
    use HasFactory;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'gauge',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'gauge' => Upper::class,
        'created_at' => 'datetime:Y-m-d',
        'updated_at' => 'datetime:Y-m-d',
        'deleted_at' => 'datetime:Y-m-d',
    ];

    /**
     * The "booted" method of the model.
     * 
     * @return void
     */
    protected static function booted(): void
    {
        static::deleting(function ($model) { 
            $model->designs?->each?->delete();
            $model->style_versions?->each?->delete();
        });

        static::restoring(function ($model) { 
            $model->designs()
                ->withTrashed()
                ->where('deleted_at', '>=', $model->deleted_at)
                ->each(function ($item, $key) { 
                    $item->restore(); 
                });
            $model->style_versions()
                ->withTrashed()
                ->where('deleted_at', '>=', $model->deleted_at)
                ->each(function ($item, $key) { 
                    $item->restore(); 
                });
        });

        static::addGlobalScope('order', function (Builder $builder) {
            $builder->orderBy('gauge');
        });
    }

    /**
     * Get the designs for this gauge.
     * 
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function designs()
    {
        return $this->hasMany(Design::class);
    }

    /**
     * Get the style versions for this gauge.
     * 
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function style_versions()
    {
        return $this->hasMany(StyleVersions::class);
    }

    /**
     * Scope a query to search for gauges.
     * 
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @param string $value
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeSearch($query, string $value)
    {
        return $query->where('gauge', 'like', "%{$value}%");
    }
}
