<x-app-layout>
    <x-slot name="header">
        <div class="flex items-center justify-between">
            <h2 class="font-semibold text-xl text-gray-800 leading-tight">
                {{ __('Library') }}
            </h2>
            <a href="{{ route('library.index') }}" class="text-indigo-600 hover:text-indigo-900">
                ← Back to Library
            </a>
        </div>
    </x-slot>

    <div class="py-12">
        <div class="max-w-4xl mx-auto sm:px-6 lg:px-8">
            <div class="bg-white overflow-hidden shadow-sm sm:rounded-lg">
                <div class="p-8">
                    <!-- Track Header -->
                    <div class="mb-8">
                        <div class="flex items-center gap-4 mb-4">
                            <span class="inline-flex items-center px-3 py-1 rounded-full text-sm font-medium bg-indigo-100 text-indigo-800" id="category-badge">
                                Loading...
                            </span>
                            <span class="text-sm text-gray-500" id="duration-badge">
                                Loading...
                            </span>
                        </div>
                        <h1 class="text-3xl font-bold text-gray-900 mb-4" id="track-title">
                            Loading...
                        </h1>
                        <p class="text-lg text-gray-600" id="track-description">
                            Loading...
                        </p>
                    </div>

                    <!-- Audio Player -->
                    <div class="mb-8">
                        <div class="bg-gray-50 rounded-lg p-6">
                            <div class="flex items-center justify-center mb-4">
                                <button id="play-button" class="w-16 h-16 bg-indigo-600 hover:bg-indigo-700 rounded-full flex items-center justify-center text-white transition-colors">
                                    <svg id="play-icon" class="w-8 h-8" fill="currentColor" viewBox="0 0 20 20">
                                        <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zM9.555 7.168A1 1 0 008 8v4a1 1 0 001.555.832l3-2a1 1 0 000-1.664l-3-2z" clip-rule="evenodd" />
                                    </svg>
                                    <svg id="pause-icon" class="w-8 h-8 hidden" fill="currentColor" viewBox="0 0 20 20">
                                        <path fill-rule="evenodd" d="M18 10a8 8 0 11-16 0 8 8 0 0116 0zM7 8a1 1 0 00-1 1v2a1 1 0 102 0V9a1 1 0 00-1-1zm4 0a1 1 0 00-1 1v2a1 1 0 102 0V9a1 1 0 00-1-1z" clip-rule="evenodd" />
                                    </svg>
                                </button>
                            </div>
                            
                            <!-- Progress Bar -->
                            <div class="mb-4">
                                <div class="flex justify-between text-sm text-gray-500 mb-2">
                                    <span id="current-time">0:00</span>
                                    <span id="total-time">0:00</span>
                                </div>
                                <div class="w-full bg-gray-200 rounded-full h-2">
                                    <div id="progress-bar" class="bg-indigo-600 h-2 rounded-full transition-all duration-300" style="width: 0%"></div>
                                </div>
                            </div>

                            <!-- Audio Element -->
                            <audio id="audio-player" preload="metadata">
                                <source id="audio-source" src="" type="audio/mpeg">
                                Your browser does not support the audio element.
                            </audio>
                        </div>
                    </div>

                    <!-- Tags -->
                    <div class="mb-8">
                        <h3 class="text-lg font-semibold text-gray-900 mb-4">Tags</h3>
                        <div class="flex flex-wrap gap-2" id="tags-container">
                            <span class="text-gray-400">Loading...</span>
                        </div>
                    </div>

                    <!-- Transcript Section -->
                    <div class="border-t pt-8">
                        <h3 class="text-lg font-semibold text-gray-900 mb-4">Transcript</h3>
                        <div class="bg-gray-50 rounded-lg p-6">
                            <div id="transcript-content" class="text-gray-700 leading-relaxed">
                                <p class="text-gray-400">Loading transcript...</p>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script>
        document.addEventListener('DOMContentLoaded', function() {
            const trackId = {{ $track }};
            let audioPlayer = null;
            let isPlaying = false;

            // Load track data
            fetchTrackData(trackId);

            function fetchTrackData(id) {
                fetch(`/api/v1/library/tracks/${id}`)
                    .then(response => response.json())
                    .then(data => {
                        if (data.data) {
                            displayTrack(data.data);
                        }
                    })
                    .catch(error => {
                        console.error('Error loading track:', error);
                        showError('Failed to load track data');
                    });
            }

            function displayTrack(track) {
                // Update UI elements
                document.getElementById('track-title').textContent = track.title;
                document.getElementById('duration-badge').textContent = formatDuration(track.duration_sec);
                document.getElementById('total-time').textContent = formatDuration(track.duration_sec);
                
                // Set audio source
                if (track.audio_url) {
                    document.getElementById('audio-source').src = track.audio_url;
                    audioPlayer = document.getElementById('audio-player');
                    setupAudioPlayer();
                } else {
                    showError('Audio not available for this track');
                }

                // Display tags
                if (track.tags && track.tags.length > 0) {
                    const tagsContainer = document.getElementById('tags-container');
                    tagsContainer.innerHTML = track.tags.map(tag => 
                        `<span class="inline-flex items-center px-3 py-1 rounded-full text-sm font-medium bg-gray-100 text-gray-800">${tag}</span>`
                    ).join('');
                }

                // Load transcript if available
                if (track.transcript_url) {
                    loadTranscript(track.transcript_url);
                }
            }

            function setupAudioPlayer() {
                if (!audioPlayer) return;

                // Play/Pause button
                document.getElementById('play-button').addEventListener('click', togglePlay);

                // Update progress bar
                audioPlayer.addEventListener('timeupdate', updateProgress);
                audioPlayer.addEventListener('loadedmetadata', () => {
                    document.getElementById('total-time').textContent = formatDuration(audioPlayer.duration);
                });

                // Handle audio end
                audioPlayer.addEventListener('ended', () => {
                    isPlaying = false;
                    updatePlayButton();
                });
            }

            function togglePlay() {
                if (!audioPlayer) return;

                if (isPlaying) {
                    audioPlayer.pause();
                } else {
                    audioPlayer.play();
                }
                isPlaying = !isPlaying;
                updatePlayButton();
            }

            function updatePlayButton() {
                const playIcon = document.getElementById('play-icon');
                const pauseIcon = document.getElementById('pause-icon');
                
                if (isPlaying) {
                    playIcon.classList.add('hidden');
                    pauseIcon.classList.remove('hidden');
                } else {
                    playIcon.classList.remove('hidden');
                    pauseIcon.classList.add('hidden');
                }
            }

            function updateProgress() {
                if (!audioPlayer) return;
                
                const progress = (audioPlayer.currentTime / audioPlayer.duration) * 100;
                document.getElementById('progress-bar').style.width = progress + '%';
                document.getElementById('current-time').textContent = formatDuration(audioPlayer.currentTime);
            }

            function formatDuration(seconds) {
                if (!seconds) return '0:00';
                const mins = Math.floor(seconds / 60);
                const secs = Math.floor(seconds % 60);
                return `${mins}:${secs.toString().padStart(2, '0')}`;
            }

            function loadTranscript(url) {
                fetch(url)
                    .then(response => response.text())
                    .then(text => {
                        document.getElementById('transcript-content').innerHTML = 
                            `<p class="whitespace-pre-wrap">${text}</p>`;
                    })
                    .catch(error => {
                        console.error('Error loading transcript:', error);
                        document.getElementById('transcript-content').innerHTML = 
                            '<p class="text-gray-400">Transcript not available</p>';
                    });
            }

            function showError(message) {
                // Simple error display - could be enhanced with a toast notification
                console.error(message);
            }
        });
    </script>
</x-app-layout>
