<?php

namespace App\Http\Livewire\Production\Reports;

use Livewire\Component;
use App\Models\Shipment;
use Livewire\Attributes\On;
use App\Models\ShipmentLine;
use App\Models\ShipmentLineSizes;
use Livewire\Attributes\Reactive;
use Livewire\Attributes\Renderless;

class ShipmentScheduleRow extends Component
{
    public array $chunk;
    public $columns;
    public $drops;

    public function mount($chunk, $selected){
        $this->chunk = array_values($chunk);
        $this->selected = [];
        foreach($this->chunk as $sl){
            $this->selected[$sl['id']] = $selected[$sl['id']] ?? FALSE;
        }
        // dd($this->selected, $this->chunk);
    }

    #[Renderless]
    #[On('refreshRows')]
    public function refreshIfModified($modifiedIds)
    {
        if (!empty(array_intersect(collect($this->chunk)->pluck('id')->toArray(), json_decode($modifiedIds)))) {
            $this->render();
        }
    }

    public function render()
    {
        $this->drops = $this->getDrops();
        return view('livewire.production.reports.shipment-schedule-row');
    }

    #[On('updateQty')]
    public function updateQty($sizeId, $value)
    {
        $this->updateShipmentLineSize($sizeId, 'qty', $value);
    }
    #[On('updateShippedQty')]
    public function updateShippedQty($sizeId, $value)
    {
        $this->updateShipmentLineSize($sizeId, 'shipped_qty', $value);
    }

    public function updateShipmentComplete($shipmentLineId, $value)
    {
        $shipmentLine = ShipmentLine::find($shipmentLineId);
        if ($shipmentLine) {
            $shipmentLine->complete = $value;
            $shipmentLine->save();
        }
    }

    public function updateTruck($shipmentLineId, $truckId)
    {
        $shipmentLine = ShipmentLine::find($shipmentLineId);
        if ($shipmentLine) {
            $shipmentLine->shipment_id = $truckId ?: null;
            $shipmentLine->save();
        }
    }

    public function getAvailableTrucks()
    {
        try {
            // First try to get trucks with transporters
            $trucks = \DB::table('shipments as sh')
                ->leftJoin('suppliers as tr', 'sh.transporter_id', '=', 'tr.id')
                ->select('sh.id', 'tr.name as transporter_name')
                ->where('tr.type', 'transporters')
                ->whereNotNull('sh.id')
                ->orderBy('sh.id')
                ->get()
                ->toArray();
            
            // If no trucks with transporters, just get all shipments
            if (empty($trucks)) {
                $trucks = \DB::table('shipments')
                    ->select('id')
                    ->whereNotNull('id')
                    ->orderBy('id')
                    ->get()
                    ->map(function($item) {
                        return ['id' => $item->id, 'transporter_name' => null];
                    })
                    ->toArray();
            }
            
            \Log::info('Available trucks:', $trucks);
            return $trucks;
        } catch (\Exception $e) {
            \Log::error('Error getting trucks:', ['error' => $e->getMessage()]);
            // Return some dummy data for testing
            return [
                ['id' => 1, 'transporter_name' => 'Test Transporter 1'],
                ['id' => 2, 'transporter_name' => 'Test Transporter 2'],
            ];
        }
    }

    /**
     * Update shipment line size using Eloquent to ensure events are triggered.
     */
    private function updateShipmentLineSize($sizeId, $field, $value)
    {
        $size = ShipmentLineSizes::find($sizeId);
        if ($size) {
            $size->$field = $value;
            $size->save();
        }
    }

    /**
     * Update shipment line using Eloquent to ensure events are triggered.
     */
    private function updateShipmentLine($lineId, $field, $value)
    {
        $line = ShipmentLine::find($lineId);
        if ($line) {
            $line->$field = $value;
            $line->save();
        }
    }

    /**
     * Update shipment using Eloquent to ensure events are triggered.
     */
    private function updateShipment($shipmentId, $field, $value)
    {
        $shipment = Shipment::find($shipmentId);
        if ($shipment) {
            $shipment->$field = $value;
            $shipment->save();
        }
    }

    public array $selected = [];

    #[On('select-all')]
    public function selectAll(){
        foreach($this->chunk as $sl){
            $this->selected[$sl['id']] = true;
        }
    }
    #[On('clear-selected')]
    public function clearSelected(){
        foreach($this->chunk as $sl){
            $this->selected[$sl['id']] = false;
        }
    }

    public function updateFlagIssue($id){
        $line = ShipmentLine::find($id);
        $line->update(['flag_issue' => !$line->flag_issue]);
    }



    public function updated($property, $value)
    {
        // Extract parts of the property name
        $parts = explode('.', $property);

        if ($parts[0] === 'drops') {
            $dropIndex = $parts[1];
            // Retrieve the actual ShipmentLine ID
            $dropId = $this->drops[$dropIndex]['id'] ?? null;

            // Handle shipment_line_sizes updates
            if (isset($parts[2]) && $parts[2] === 'shipment_line_sizes') {
                $sizeIndex = $parts[3]; // Array index, NOT the ID
                $field = $parts[4] ?? null; // Field being updated

                // Get the actual ID from the array
                $sizeId = $this->drops[$dropIndex]['shipment_line_sizes'][$sizeIndex]['id'] ?? null;

                if ($sizeId && in_array($field, ['qty', 'shipped_qty'])) {
                    $this->updateShipmentLineSize($sizeId, $field, $value);
                    return;
                }
            }

            // Handle shipments updates
            if (isset($parts[2]) && $parts[2] === 'shipments') {
                $shipmentField = $parts[3] ?? null;

                if ($shipmentField === 'transporter_invoice') {
                    $shipmentId = $this->drops[$dropIndex]['shipments']['id'] ?? null;
                    if ($shipmentId) {
                        $this->updateShipment($shipmentId, 'transporter_invoice', $value);
                    }
                    return;
                }
            }

            if (in_array($parts[2] ?? null, ['notes', 'fn_notes', 'rt_invoice', 'factory_invoice', 'customs_agency_invoice', 'invoiced', 'net_weight', 'gross_weight', 'no_cartons'])) {
                $this->updateShipmentLine($dropId, $parts[2], $value);
                return;
            }
        }
    }

    public function getDrops(){
        $ids = collect($this->chunk)->pluck('id')->toArray();
        
        if (empty($ids)) {
            return [];
        }

        // Run the main query
        $results = $this->runPriceQuery($ids);

        // Group and structure the data
        $drops = $this->structureDropsData($results);
        
        return $drops;
    }

    /**
     * Run the main SQL query with all joins including price data.
     */
    private function runPriceQuery($ids)
    {
        return \DB::table('shipment_lines as sl')
            // Join customer order lines
            ->join('customer_order_lines as col', 'sl.customer_order_lines_id', '=', 'col.id')
            // Join customer orders
            ->join('customer_orders as co', 'col.customer_orders_id', '=', 'co.id')
            // Join customers
            ->join('customers as c', 'co.customers_id', '=', 'c.id')
            // Join seasons
            ->join('seasons as s', 'co.seasons_id', '=', 's.id')
            // Join colourways
            ->join('colourways as cw', 'col.colourways_id', '=', 'cw.id')
            // Join style versions
            ->join('style_versions as sv', 'cw.style_versions_id', '=', 'sv.id')
            // Join styles
            ->join('styles as st', 'sv.styles_id', '=', 'st.id')
            // Join designs
            ->join('designs as d', 'st.designs_id', '=', 'd.id')
            // Join factories (suppliers)
            ->join('suppliers as f', 'sv.factory_id', '=', 'f.id')
            // Join customer order line quantities
            ->join('customer_order_line_quantities as colq', 'col.id', '=', 'colq.customer_order_lines_id')
            // Join sizes
            ->join('sizes as sz', 'colq.sizes_id', '=', 'sz.id')
            // Join shipment line sizes
            ->leftJoin('shipment_line_sizes as sls', function($join) {
                $join->on('sl.id', '=', 'sls.shipment_line_id')
                     ->on('colq.sizes_id', '=', 'sls.sizes_id');
            })
            // Join shipments
            ->leftJoin('shipments as sh', 'sl.shipment_id', '=', 'sh.id')
            // Join transporters
            ->leftJoin('suppliers as tr', 'sh.transporter_id', '=', 'tr.id')
            // Join price resolutions
            ->leftJoin('price_resolutions as pr', function($join) {
                $join->on('pr.colourways_id', '=', 'cw.id')
                     ->on('pr.sizes_id', '=', 'colq.sizes_id')
                     ->whereNull('pr.phase_id')
                     ->on('pr.season_id', '=', 'co.seasons_id')
                     ->whereNotNull('pr.fresh_at');
            })
            // Join shipment samples (type 7) - only get the latest one
            ->leftJoin('samples as ss', function($join) {
                $join->on('ss.colourways_id', '=', 'cw.id')
                     ->where('ss.sample_types_id', '=', 7)
                     ->whereNull('ss.deleted_at')
                     ->whereRaw('ss.created_at = (SELECT MAX(created_at) FROM samples WHERE colourways_id = cw.id AND sample_types_id = 7 AND deleted_at IS NULL)');
            })
            // Join sealer samples (type 3) - only get the latest one
            ->leftJoin('samples as ses', function($join) {
                $join->on('ses.colourways_id', '=', 'cw.id')
                     ->where('ses.sample_types_id', '=', 3)
                     ->whereNull('ses.deleted_at')
                     ->whereRaw('ses.created_at = (SELECT MAX(created_at) FROM samples WHERE colourways_id = cw.id AND sample_types_id = 3 AND deleted_at IS NULL)');
            })
            // Join total cache for shipment line totals
            ->leftJoin('total_cache as tc', function($join) {
                $join->on('tc.entity_id', '=', 'sl.id')
                     ->where('tc.entity_type', '=', 'shipment_line')
                     ->where('tc.cache_key', '=', 'prices')
                     ->whereNotNull('tc.fresh_at');
            })
            ->whereIn('sl.id', $ids)
            ->select([
                // Shipment line data
                'sl.id as shipment_line_id', 'sl.shipment_id', 'sl.complete', 'sl.flag_issue', 'sl.exfty',
                'sl.notes', 'sl.fn_notes', 'sl.rt_invoice', 'sl.factory_invoice', 'sl.customs_agency_invoice',
                'sl.invoiced', 'sl.net_weight', 'sl.gross_weight', 'sl.no_cartons', 'sl.collection_date',
                'sl.shipment_status', 'sl.shipment_comments', 'sl.sealer_status', 'sl.sealer_comments',

                // Customer order line data
                'col.id as customer_order_line_id', 'col.factory_cust_date', 'col.wh_cust_date', 'col.cancelled as col_cancelled',

                // Customer order data
                'co.id as customer_order_id', 'co.order_date', 'co.customer_po', 'co.incoterms',

                // Customer data
                'c.id as customer_id', 'c.name as customer_name', 'c.currency as customer_currency',
                'c.settings as customer_settings', 'c.samples_required',

                // Season data
                's.id as season_id', 's.description as season_description',

                // Colourway data
                'cw.id as colourway_id', 'cw.name as colourway_name', 'cw.img_thumb',
                'cw.cancelled as colourway_cancelled', 'cw.colour_type',

                // Style version data
                'sv.id as style_version_id', 'sv.name as style_version_name',

                // Style data
                'st.id as style_id', 'st.customer_ref', 'st.cancelled as style_cancelled', 'st.carryover',

                // Design data
                'd.id as design_id', 'd.description as design_description',

                // Factory data
                'f.id as factory_id', 'f.name as factory_name', 'f.currency as factory_currency',

                // Customer order line quantity data
                'colq.id as order_line_quantity_id', 'colq.qty as ordered_qty', 'colq.SKU', 'colq.barcode',

                // Size data
                'sz.id as size_id', 'sz.name as size_name', 'sz.order as size_order',

                // Shipment line size data
                'sls.id as shipment_line_size_id', 'sls.qty', 'sls.shipped_qty',

                // Shipment data
                'sh.id as shipment_id', 'sh.cost as shipment_cost', 'sh.organiser',
                'sh.customs_submitted_text', 'sh.transporter_invoice',

                // Transporter data
                'tr.id as transporter_id', 'tr.name as transporter_name', 'tr.currency as transporter_currency',

                // Price resolution data
                'pr.quote', 'pr.cmt', 'pr.quote_status', 'pr.cmt_status', 'pr.quote_base', 'pr.cmt_base',
                'pr.discount_price', 'pr.discount_price_base', 'pr.subtotal', 'pr.subtotal_base',
                'pr.gmt_trans_base', 'pr.yarn_value_euro',

                // Cached total data
                'tc.cached_data as cached_totals',

                // Shipment sample data (type 7)
                'ss.id as shipment_sample_id', 'ss.status as shipment_sample_status',
                'ss.comments as shipment_sample_comments', 'ss.date_sent as shipment_sample_date_sent',

                // Sealer sample data (type 3)
                'ses.id as sealer_sample_id', 'ses.status as sealer_sample_status',
                'ses.comments as sealer_sample_comments', 'ses.date_sent as sealer_sample_date_sent'
            ])
            ->orderByRaw("FIELD(sl.id, " . implode(',', $ids) . ")")
            ->orderBy('sz.order')
            ->get();
    }

    /**
     * Structure the flat SQL results into the nested format expected by the template.
     * 
     * This method now relies entirely on cached totals for pricing calculations,
     * which are automatically updated when quantities or prices change.
     */
    private function structureDropsData($results)
    {
        $drops = [];
        
        // Store raw results for cached totals lookup
        $rawResults = $results->toArray();
        
        // Pre-build cached totals lookup for performance
        $cachedTotalsLookup = [];
        foreach ($rawResults as $rawRow) {
            if ($rawRow->cached_totals) {
                $cachedTotalsLookup[$rawRow->shipment_line_id] = json_decode($rawRow->cached_totals, true);
            }
        }
        
        foreach ($results as $row) {
            $shipmentLineId = $row->shipment_line_id;
            
            // Initialize drop if not exists
            if (!isset($drops[$shipmentLineId])) {
                $drops[$shipmentLineId] = [
                    'id' => $row->shipment_line_id,
                    'shipment_id' => $row->shipment_id,
                    'complete' => $row->complete,
                    'flag_issue' => $row->flag_issue,
                    'exfty' => $row->exfty,
                    'notes' => $row->notes,
                    'fn_notes' => $row->fn_notes,
                    'rt_invoice' => $row->rt_invoice,
                    'factory_invoice' => $row->factory_invoice,
                    'customs_agency_invoice' => $row->customs_agency_invoice,
                    'invoiced' => $row->invoiced,
                    'net_weight' => $row->net_weight,
                    'gross_weight' => $row->gross_weight,
                    'no_cartons' => $row->no_cartons,
                    'collection_date' => $row->collection_date,
                    
                    'customer_order_lines' => [
                        'id' => $row->customer_order_line_id,
                        'customer_orders_id' => $row->customer_order_id,
                        'factory_cust_date' => $row->factory_cust_date,
                        'wh_cust_date' => $row->wh_cust_date,
                        'cancelled' => $row->col_cancelled,
                        
                        'customer_orders' => [
                            'id' => $row->customer_order_id,
                            'order_date' => $row->order_date,
                            'customer_po' => $row->customer_po,
                            'incoterms' => $row->incoterms,
                            
                            'seasons' => [
                                'id' => $row->season_id,
                                'description' => $row->season_description,
                            ],
                            
                            'customers' => [
                                'id' => $row->customer_id,
                                'name' => $row->customer_name,
                                'currency' => $row->customer_currency,
                                'settings' => $row->customer_settings,
                                'samples_required' => $row->samples_required,
                            ],
                        ],
                        
                        'colourways' => [
                            'id' => $row->colourway_id,
                            'name' => $row->colourway_name,
                            'img_thumb' => $row->img_thumb,
                            'thumb_url' => $row->img_thumb ? url(asset('storage/' . $row->img_thumb)) : null,
                            'cancelled' => $row->colourway_cancelled,
                            'colour_type' => $row->colour_type,
                            
                            'style_versions' => [
                                'id' => $row->style_version_id,
                                'name' => $row->style_version_name,
                                'styles_id' => $row->style_id,
                                
                                'factories' => [
                                    'id' => $row->factory_id,
                                    'name' => $row->factory_name,
                                    'currency' => $row->factory_currency,
                                ],
                                
                                'styles' => [
                                    'id' => $row->style_id,
                                    'customer_ref' => $row->customer_ref,
                                    'cancelled' => $row->style_cancelled,
                                    'designs_id' => $row->design_id,
                                    
                                    'designs' => [
                                        'id' => $row->design_id,
                                        'description' => $row->design_description,
                                    ],
                                ],
                            ],
                        ],
                        
                        'customer_order_line_quantities' => [],
                    ],
                    
                    'shipment_line_sizes' => [],
                    
                    'shipments' => $row->shipment_id ? [
                        'id' => $row->shipment_id,
                        'cost' => $row->shipment_cost,
                        'organiser' => $row->organiser,
                        'customs_submitted_text' => $row->customs_submitted_text,
                        'transporter_invoice' => $row->transporter_invoice,
                        
                        'transporters' => $row->transporter_id ? [
                            'id' => $row->transporter_id,
                            'name' => $row->transporter_name,
                            'currency' => $row->transporter_currency,
                        ] : null,
                    ] : null,
                    
                    // Sample data
                    'shipment_sample' => $this->buildSampleData($row, 'shipment'),
                    'sealer_sample' => $this->buildSampleData($row, 'sealer'),
                ];
            }
            
            // Add customer order line quantity
            $quantityId = $row->order_line_quantity_id;
            if (!isset($drops[$shipmentLineId]['customer_order_lines']['customer_order_line_quantities'][$quantityId])) {
                $drops[$shipmentLineId]['customer_order_lines']['customer_order_line_quantities'][$quantityId] = [
                    'id' => $row->order_line_quantity_id,
                    'qty' => $row->ordered_qty,
                    'SKU' => $row->SKU,
                    'barcode' => $row->barcode,
                    
                    'sizes' => [
                        'id' => $row->size_id,
                        'name' => $row->size_name,
                        'order' => $row->size_order,
                    ],
                    
                    'price_model_data' => ($row->quote !== null || $row->cmt !== null) ? [
                        'quote' => $row->quote,
                        'cmt' => $row->cmt,
                        'quote_status' => $row->quote_status,
                        'cmt_status' => $row->cmt_status,
                        'quote_base' => $row->quote_base,
                        'cmt_base' => $row->cmt_base,
                        'discount_price' => $row->discount_price,
                        'discount_price_base' => $row->discount_price_base,
                        'subtotal' => $row->subtotal,
                        'subtotal_base' => $row->subtotal_base,
                        'transport_budget' => $row->gmt_trans_base,
                        'transport_budget_base' => $row->gmt_trans_base,
                        'yarn_cost_kg' => $row->yarn_value_euro,
                    ] : null,
                ];
            }
            
            // Add shipment line size if exists
            if ($row->shipment_line_size_id && !isset($drops[$shipmentLineId]['shipment_line_sizes'][$row->shipment_line_size_id])) {
                $drops[$shipmentLineId]['shipment_line_sizes'][$row->shipment_line_size_id] = [
                    'id' => $row->shipment_line_size_id,
                    'qty' => $row->qty,
                    'shipped_qty' => $row->shipped_qty,
                    'sizes_id' => $row->size_id,
                    
                    'sizes' => [
                        'id' => $row->size_id,
                        'name' => $row->size_name,
                        'order' => $row->size_order,
                    ],
                ];
            }
        }
        
        // Convert to indexed arrays and calculate totals
        $structuredDrops = [];
        foreach ($drops as $shipmentLineId => $drop) {
            // Convert quantities to indexed array
            $drop['customer_order_lines']['customer_order_line_quantities'] = array_values($drop['customer_order_lines']['customer_order_line_quantities']);
            
            // Convert shipment line sizes to indexed array
            $drop['shipment_line_sizes'] = array_values($drop['shipment_line_sizes']);
            
            // Calculate totals
            $drop['total_pieces'] = array_sum(array_column($drop['customer_order_lines']['customer_order_line_quantities'], 'qty'));
            $drop['total_pieces_shipped'] = array_sum(array_column($drop['shipment_line_sizes'], 'shipped_qty'));
            
            // Get cached totals from pre-built lookup
            $cachedTotals = $cachedTotalsLookup[$shipmentLineId] ?? null;
            
            // Use cached totals (automatically populated when quantities/prices change)
            $drop['total_prices'] = [
                'quote' => $cachedTotals['quote'] ?? 0,
                'cmt' => $cachedTotals['cmt'] ?? 0,
                'subtotal' => $cachedTotals['subtotal'] ?? 0,
                'transport_budget' => $cachedTotals['transport_budget'] ?? 0,
            ];
            
            $structuredDrops[] = $drop;
        }
        
        return $structuredDrops;
    }


    /**
     * Build sample data structure based on the row data and sample type.
     */
    private function buildSampleData($row, $sampleType)
    {
        $sampleIdField = $sampleType . '_sample_id';
        $sampleStatusField = $sampleType . '_sample_status';
        $sampleCommentsField = $sampleType . '_sample_comments';
        $sampleDateSentField = $sampleType . '_sample_date_sent';
        $overrideStatusField = $sampleType . '_status';
        $overrideCommentsField = $sampleType . '_comments';
        
        // Check if we have a sample record OR an override status
        if ($row->$sampleIdField || $row->$overrideStatusField) {
            return [
                'status' => $row->$sampleStatusField ?? $row->$overrideStatusField,
                'id' => $row->$sampleIdField,
                'comments' => $row->$sampleCommentsField ?? $row->$overrideCommentsField,
                'date_sent' => $row->$sampleDateSentField,
                'note' => $row->$sampleIdField ? null : '(SS)',
            ];
        }
        
        // No sample record and no override - check customer settings and carryover status
        $sampleTypeId = $sampleType === 'shipment' ? 7 : 3;
        $samplesRequired = json_decode($row->samples_required, true);
        $sampleRequired = $samplesRequired[$sampleTypeId] ?? false;
        
        // For carryover samples, we would need to check customer settings
        // For now, we'll use the basic logic
        $status = $row->carryover 
            ? 'Not Required' // In full implementation, would check carryover-samples-required
            : ($sampleRequired ? 'Required' : 'Not Required');
        
        return [
            'status' => $status,
            'id' => null,
            'comments' => null,
            'date_sent' => null,
            'note' => $row->carryover ? '(Carryover)' : null,
        ];
    }

    public function notesColsCount()
    {
        $i = 0;
        if ($this->columns['Season']) $i++;
        if ($this->columns['Factory']) $i++;
        if ($this->columns['Customer']) $i++;
        if ($this->columns['Order No']) $i++;
        if ($this->columns['Style']) $i++;
        if ($this->columns['Description']) $i++;
        if ($this->columns['Colour']) $i++;
        return $i;
    }

    public function fnNotesColsCount()
    {
        $i = 0;
        if ($this->columns['RT Invoice']) $i++;
        if ($this->columns['Factory Invoice']) $i++;
        if ($this->columns['Customs Invoice']) $i++;
        if ($this->columns['Transporter Invoice']) $i++;
        if ($this->columns['Finance Complete']) $i++;
        return $i;
    }
}
