<?php

namespace App\Livewire\Manage;

use Livewire\Component;
use App\Models\Statement;
use App\Models\StatementTransaction;
use App\Models\Receipt;
use App\Models\ReceiptLine;
use App\Models\Account;
use App\Models\Department;
use Livewire\Attributes\On;
use App\Jobs\MatchReceipt;
use App\Jobs\ProcessReceipt;
use App\Services\ReceiptMergingService;
use App\Jobs\ProcessMergedReceipt;

class StatementReview extends Component
{
    public ?int $statementId = null;
    public bool $matchingRequested = false;
    public ?int $selectedReceiptId = null;
    public array $editReceipt = [];
    public array $transactions = [];
    public array $splits = [];
    public array $expanded = [];
    public array $selectedReceipts = [];
    public bool $showMergeButton = false;

    public function mount(?int $statementId = null)
    {
        // If no statement is selected, show all transactions by default
        // For now, always show all transactions to see all matches
        $this->statementId = null;
    }

    #[On('select-statement')]
    public function onSelectStatement(int $statementId): void
    {
        $this->statementId = $statementId;
    }

    public function showAllTransactions(): void
    {
        $this->statementId = null;
    }

    public function render()
    {
        $statement = $this->statementId ? Statement::with('transactions.matches.receipt.lines')->find($this->statementId) : null;
        $unmatchedReceiptsQuery = Receipt::doesntHave('matches');
        $matchedReceiptsQuery = Receipt::whereHas('matches');
        // Remove date filtering to show all receipts regardless of statement period
        $unmatchedReceipts = $unmatchedReceiptsQuery->latest()->limit(20)->get();
        $matchedReceipts = $matchedReceiptsQuery->latest()->limit(20)->get();
        $unmatchedReceiptsCount = (clone $unmatchedReceiptsQuery)->count();

        $unmatchedTransactionsQuery = StatementTransaction::when($this->statementId, fn($q) => $q->where('statement_id', $this->statementId))
            ->doesntHave('matches');
        $unmatchedTransactions = (clone $unmatchedTransactionsQuery)->latest()->limit(50)->get();
        $unmatchedTransactionsCount = (clone $unmatchedTransactionsQuery)->count();

        // Get all transactions for the lines table (matched and unmatched)
        // Prioritize matched transactions first, then by latest date
        // Show all transactions if no specific statement is selected
        $allTransactionsQuery = StatementTransaction::when($this->statementId, fn($q) => $q->where('statement_id', $this->statementId));
        $allTransactions = (clone $allTransactionsQuery)
            ->with('matches.receipt.lines')
            ->orderBy('has_receipt', 'desc') // Show matched transactions first
            ->orderBy('created_at', 'desc')  // Then by latest date
            ->get();

        // Initialize transactions array with current values from database
        foreach ($allTransactions as $txn) {
            if (!array_key_exists($txn->id, $this->transactions)) {
                $this->transactions[$txn->id] = [
                    'department_id' => $txn->department_id,
                    'account_id' => $txn->account_id,
                ];
            }
        }

        // Initialize splits state from transaction meta if not already present
        foreach ($allTransactions as $txn) {
            $meta = is_array($txn->meta) ? $txn->meta : (empty($txn->meta) ? [] : (json_decode($txn->meta, true) ?: []));
            if (!array_key_exists($txn->id, $this->splits)) {
                $this->splits[$txn->id] = isset($meta['splits']) && is_array($meta['splits']) ? $meta['splits'] : [];
            }
            if (!array_key_exists($txn->id, $this->expanded)) {
                $this->expanded[$txn->id] = false;
            }
        }

        // Preload selectable data to avoid N+1 in the blade
        $departments = Department::select('id','name','code')->orderBy('name')->get();
        $accounts = Account::select('id','name','code')->orderBy('name')->get();

        // Load selected receipt once (with lines) to avoid querying in the blade
        $selectedReceipt = $this->selectedReceiptId ? Receipt::with('lines')->find($this->selectedReceiptId) : null;

        return view('livewire.manage.statement-review', compact(
            'statement',
            'unmatchedReceipts',
            'matchedReceipts',
            'unmatchedTransactions',
            'unmatchedReceiptsCount',
            'unmatchedTransactionsCount',
            'allTransactions',
            'departments',
            'accounts',
            'selectedReceipt',
        ));
    }

    public function runMatching(): void
    {
        $query = Receipt::doesntHave('matches');
        // Remove date filtering to match all unmatched receipts
        $query->limit(100)->pluck('id')->each(function ($id) {
            MatchReceipt::dispatch((int)$id);
        });
        $this->matchingRequested = true;
        session()->flash('status', 'Matching started for unmatched receipts. This view will refresh every 5s.');
    }

    public function selectReceipt(int $receiptId): void
    {
        $this->selectedReceiptId = $receiptId;
        $receipt = Receipt::with('lines')->find($receiptId);
        if ($receipt) {
            $this->editReceipt = [
                'id' => $receipt->id,
                'total_amount' => number_format((float) $receipt->total_amount, 2, '.', ''),
                'currency' => $receipt->currency,
                'lines' => $receipt->lines->map(function ($line) {
                    return [
                        'id' => $line->id,
                        'description' => $line->description,
                        'quantity' => (float) $line->quantity,
                        'unit_price' => number_format((float) $line->unit_price, 2, '.', ''),
                        'line_total' => number_format((float) $line->line_total, 2, '.', ''),
                    ];
                })->toArray(),
            ];
        }
    }

    public function closeModal(): void
    {
        $this->selectedReceiptId = null;
    }

    public function reprocessReceipt(int $receiptId): void
    {
        // Trigger a fresh OCR + classification pass for this receipt
        ProcessReceipt::dispatch($receiptId)->onQueue('ocr');
        $this->matchingRequested = true; // so the header poll/refresh continues
        session()->flash('status', 'OCR re-run started for receipt #'.$receiptId.'.');
    }

    public function matchReceipt(int $receiptId): void
    {
        MatchReceipt::dispatch($receiptId)->onQueue('matching');
        $this->matchingRequested = true; // so the header poll/refresh continues
        session()->flash('status', 'Matching started for receipt #'.$receiptId.'.');
    }

    public function updateReceiptTotal(int $receiptId, $value): void
    {
        $receipt = Receipt::find($receiptId);
        if (!$receipt) return;
        $amount = is_numeric($value) ? (float)$value : null;
        if ($amount === null) return;
        $receipt->total_amount = $amount;
        $receipt->save();
        session()->flash('status', 'Receipt total updated.');
    }

    public function updateLineAmount(int $lineId, string $field, $value): void
    {
        $allowed = ['unit_price', 'line_total'];
        if (!in_array($field, $allowed, true)) return;
        $line = ReceiptLine::find($lineId);
        if (!$line) return;
        $num = is_numeric($value) ? (float)$value : null;
        if ($num === null) return;
        $line->$field = $num;
        // if unit_price changed and quantity present, recompute line_total conservatively only when not zero
        if ($field === 'unit_price' && $line->quantity) {
            $line->line_total = round($line->quantity * $line->unit_price, 2);
        }
        $line->save();
        session()->flash('status', 'Line updated.');
    }

    public function saveReceiptEdits(): void
    {
        if (!$this->selectedReceiptId || empty($this->editReceipt)) return;
        $receipt = Receipt::with('lines')->find($this->selectedReceiptId);
        if (!$receipt) return;

        $total = round((float) ($this->editReceipt['total_amount'] ?? 0), 2);
        $lines = $this->editReceipt['lines'] ?? [];
        $sum = 0.0;
        foreach ($lines as $l) {
            $sum += round((float) ($l['line_total'] ?? 0), 2);
        }
        $sum = round($sum, 2);
        if (abs($sum - $total) > 0.01) {
            session()->flash('error', 'Line totals (£' . number_format($sum, 2) . ') must equal receipt total (£' . number_format($total, 2) . ').');
            return;
        }

        // Persist receipt total
        $receipt->total_amount = $total;
        $receipt->save();

        // Persist each line
        foreach ($lines as $l) {
            $line = $receipt->lines->firstWhere('id', $l['id'] ?? 0);
            if (!$line) continue;
            $line->line_total = round((float) ($l['line_total'] ?? 0), 2);
            $qty = max(1.0, (float) ($l['quantity'] ?? 1));
            // Recompute unit price from line_total to ensure consistency
            $line->unit_price = round($line->line_total / $qty, 2);
            $line->save();
        }

        // Trigger re-matching after saving edits
        MatchReceipt::dispatch($receipt->id)->onQueue('matching');
        $this->matchingRequested = true;

        // Close the modal and show success message
        $this->selectedReceiptId = null;
        $this->editReceipt = [];
        session()->flash('status', 'Receipt totals and line items saved. Re-matching started.');
    }

    public function updatedEditReceipt($value, $key): void
    {
        // Auto-update total when line items change
        if (str_starts_with($key, 'lines.') && str_contains($key, '.line_total')) {
            $this->updateTotalFromLines();
        }
    }

    private function updateTotalFromLines(): void
    {
        if (empty($this->editReceipt['lines'])) return;
        
        $sum = 0.0;
        foreach ($this->editReceipt['lines'] as $line) {
            $sum += round((float) ($line['line_total'] ?? 0), 2);
        }
        
        $this->editReceipt['total_amount'] = number_format($sum, 2, '.', '');
    }

    public function updatedTransactions($value, $key): void
    {
        // Expecting key like "transactions.{transactionId}.{field}"
        $parts = explode('.', (string) $key);
        if (count($parts) >= 3 && $parts[0] === 'transactions') {
            $transactionId = (int) $parts[1];
            $field = $parts[2];

            // Only allow specific fields to be updated from the UI
            $allowedFields = ['department_id', 'account_id'];
            if (!in_array($field, $allowedFields, true)) {
                return;
            }

            // Normalize empty values to null; cast numeric values
            $normalized = ($value === '' || $value === null) ? null : (int) $value;

            // Find the transaction and update it
            $transaction = StatementTransaction::find($transactionId);
            if ($transaction) {
                $transaction->update([$field => $normalized]);
                session()->flash('status', 'Transaction updated successfully.');
            }
        }
    }

    public function updatedSplits($value, $key): void
    {
        // Expecting key like "splits.{transactionId}.{index}.{field}"
        $parts = explode('.', (string) $key);
        if (count($parts) >= 4 && $parts[0] === 'splits') {
            $transactionId = (int) $parts[1];
            $this->persistSplits($transactionId);
            session()->flash('status', 'Split updated.');
        }
    }

    public function toggleExpand(int $transactionId): void
    {
        $this->expanded[$transactionId] = !($this->expanded[$transactionId] ?? false);
    }

    public function addSplit(int $transactionId): void
    {
        $transaction = StatementTransaction::find($transactionId);
        if (!$transaction) return;

        $this->expanded[$transactionId] = true;
        $this->splits[$transactionId] = $this->splits[$transactionId] ?? [];
        $this->splits[$transactionId][] = [
            'department_id' => $transaction->department_id,
            'account_id' => $transaction->account_id,
            'amount' => null,
            'description' => null,
            'source' => 'manual',
        ];
        $this->persistSplits($transactionId);
    }

    public function removeSplit(int $transactionId, int $index): void
    {
        if (!isset($this->splits[$transactionId][$index])) return;
        array_splice($this->splits[$transactionId], $index, 1);
        $this->persistSplits($transactionId);
    }

    public function clearSplits(int $transactionId): void
    {
        $this->splits[$transactionId] = [];
        $this->persistSplits($transactionId);
    }

    public function applyReceiptSplits(int $transactionId): void
    {
        $transaction = StatementTransaction::with('matches.receipt.lines')->find($transactionId);
        if (!$transaction) {
            session()->flash('error', 'Transaction not found.');
            return;
        }

        if ($transaction->matches->isEmpty()) {
            session()->flash('error', 'No receipts matched to this transaction.');
            return;
        }

        // Collect all receipt lines from matched receipts
        $lines = collect();
        foreach ($transaction->matches as $match) {
            if ($match->receipt) {
                // Load lines if not already loaded
                if (!$match->receipt->relationLoaded('lines')) {
                    $match->receipt->load('lines');
                }
                $lines = $lines->merge($match->receipt->lines);
            }
        }

        if ($lines->isEmpty()) {
            session()->flash('error', 'No receipt lines found for this transaction.');
            return;
        }

        // Create a split for each line item so departments can be assigned individually
        $splits = [];
        $totalAmount = 0;
        
        foreach ($lines as $line) {
            // Check if line has discount info in meta
            $meta = is_array($line->meta) ? $line->meta : (is_string($line->meta) ? json_decode($line->meta, true) : []);
            
            // Use discounted_total if available, otherwise use line_total
            if (isset($meta['discount_info']['discounted_total'])) {
                $amount = (float) $meta['discount_info']['discounted_total'];
            } else {
                $amount = (float) $line->line_total;
            }
            
            if ($amount <= 0) continue;
            
            $totalAmount += $amount;
            $splits[] = [
                'department_id' => $line->department_id ?? $transaction->department_id,
                'account_id' => $line->account_id ?? $transaction->account_id,
                'amount' => round($amount, 2),
                'description' => $line->description ?? 'Receipt line item',
                'source' => 'receipt_line',
            ];
        }

        if (empty($splits)) {
            session()->flash('error', 'No valid line items found (all have £0 amount).');
            return;
        }

        $this->splits[$transactionId] = $splits;
        $this->expanded[$transactionId] = true;
        $this->persistSplits($transactionId);
        session()->flash('status', 'Created ' . count($splits) . ' split(s) from ' . $lines->count() . ' receipt line(s) totaling £' . number_format($totalAmount, 2) . '. Assign departments as needed.');
    }

    private function persistSplits(int $transactionId): void
    {
        $transaction = StatementTransaction::find($transactionId);
        if (!$transaction) return;
        $meta = is_array($transaction->meta) ? $transaction->meta : (empty($transaction->meta) ? [] : (json_decode($transaction->meta, true) ?: []));
        $meta['splits'] = $this->splits[$transactionId] ?? [];
        $transaction->meta = $meta;
        $transaction->save();
    }

    // Receipt Merging Methods

    public function toggleReceiptSelection(int $receiptId): void
    {
        if (in_array($receiptId, $this->selectedReceipts)) {
            $this->selectedReceipts = array_filter($this->selectedReceipts, fn($id) => $id !== $receiptId);
        } else {
            $this->selectedReceipts[] = $receiptId;
        }
        
        $this->showMergeButton = count($this->selectedReceipts) >= 2;
    }

    public function mergeSelectedReceipts(ReceiptMergingService $mergingService): void
    {
        if (count($this->selectedReceipts) < 2) {
            session()->flash('error', 'Please select at least 2 receipts to merge.');
            return;
        }

        try {
            $mergedReceipt = $mergingService->mergeReceipts($this->selectedReceipts);
            
            // Dispatch job to process the merged receipt
            ProcessMergedReceipt::dispatch($mergedReceipt->id);
            
            $selectedCount = count($this->selectedReceipts);
            
            // Clear selections
            $this->selectedReceipts = [];
            $this->showMergeButton = false;
            
            session()->flash('status', "Successfully merged {$selectedCount} receipts. The merged receipt is being processed and will appear shortly.");
            
        } catch (\Exception $e) {
            session()->flash('error', 'Failed to merge receipts: ' . $e->getMessage());
        }
    }

    public function clearSelection(): void
    {
        $this->selectedReceipts = [];
        $this->showMergeButton = false;
    }

    public function isReceiptSelected(int $receiptId): bool
    {
        return in_array($receiptId, $this->selectedReceipts);
    }
}


