<?php

namespace App\Models;

use App\Casts\Currency;
use Illuminate\Support\Str;
use \Bkwld\Cloner\Cloneable;
use App\Models\BaseModel;
use OwenIt\Auditing\Contracts\Auditable;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class CustomerOrderLineQuantities extends BaseModel implements Auditable
{
    use \OwenIt\Auditing\Auditable;
    use SoftDeletes;
    use HasFactory;
    use Cloneable;

    protected $appends = ['price_model', 'cached_price_model'];

    protected $casts = [
        'price' => Currency::class,
        'created_at' => 'datetime:Y-m-d',
        'updated_at' => 'datetime:Y-m-d',
        'deleted_at' => 'datetime:Y-m-d',
    ];
    protected $fillable = [
        'customer_order_lines_id',
        'sizes_id',
        'qty',
        'SKU',
        'barcode',
        'created_at',
        'updated_at',
        'price',
        'discount',
        'commission',
        'deleted_at',

        'quote_cache',
        'cmt_cache',
        'cost_cache',

        'prices_id',

        'transport_budget_cache',
        'quote_status_cache',
        'cmt_status_cache',
    ];

    public function getPriceModelAttribute(){
        // Try to get from price resolution cache first
        $resolution = $this->getPriceResolution();
        
        if ($resolution) {
            return [
                'quote_base' => $resolution->quote_base,
                'quote' => $resolution->quote,
                'discount_price' => $resolution->discount_price,
                'cmt' => $resolution->cmt,
                'subtotal' => $resolution->subtotal,
                'subtotal_base' => $resolution->subtotal_base,
                'transport_budget' => $resolution->gmt_trans_base,
                'transport_budget_base' => $resolution->gmt_trans_base,
                'quote_status' => $resolution->quote_status,
                'cmt_status' => $resolution->cmt_status,
                'yarn_cost_kg' => $resolution->yarn_value_euro,
            ];
        }

        // Fallback to original calculation if no resolution found
        $priceInfo = $this->customer_order_lines->colourways->price($this->sizes_id, $this->customer_order_lines->phase_id);
        // dd($priceInfo, $this->sizes_id, $this->customer_order_lines->phase_id);
        if($priceInfo['status'] == 'success'){
            $price = $priceInfo['price'];
            // dd($price);
            return [
                'quote_base' => $price->quote_base,
                'quote' => $price->quote,
                'discount_price' => $price->discount_price,
                'cmt' => $price->cmt,
                'subtotal' => $price->subtotal,
                'subtotal_base' => $price->subtotal_base,
                'transport_budget' => convertCurrency($price->gmt_trans_currency, '£', $price->gmt_trans, $this->customer_order_lines->customer_orders->seasons_id),
                'transport_budget_base' => convertCurrency($price->gmt_trans_currency, '£', $price->gmt_trans, $this->customer_order_lines->customer_orders->seasons_id),
                'quote_status' => $price->quote_status,
                'cmt_status' => $price->cmt_status,
                'yarn_cost_kg' => $price->yarn_cost_kg,
            ];
        }
    }

    public function getCachedPriceModelAttribute(){
        // Only return cached price resolution data, no fallback
        $resolution = $this->getPriceResolution();
        
        if ($resolution) {
            return [
                'quote_base' => $resolution->quote_base,
                'quote' => $resolution->quote,
                'discount_price' => $resolution->discount_price,
                'cmt' => $resolution->cmt,
                'subtotal' => $resolution->subtotal,
                'subtotal_base' => $resolution->subtotal_base,
                'transport_budget' => $resolution->gmt_trans_base,
                'transport_budget_base' => $resolution->gmt_trans_base,
                'quote_status' => $resolution->quote_status,
                'cmt_status' => $resolution->cmt_status,
                'yarn_cost_kg' => $resolution->yarn_value_euro,
            ];
        }
        
        return null;
    }

    public function getTransportBudgetAttribute(){
        $resolution = $this->getPriceResolution();
        if ($resolution) {
            return $resolution->gmt_trans_base;
        }

        // Fallback to original calculation
        $priceInfo = $this->customer_order_lines->colourways->price($this->sizes_id, $this->customer_order_lines->phase_id);
        if($priceInfo['status'] == 'success'){
            $price = $priceInfo['price'];
            return $price->gmt_trans;
        }
    }

    public function getTransportBudgetBaseAttribute(){
        $resolution = $this->getPriceResolution();
        if ($resolution) {
            return $resolution->gmt_trans_base;
        }

        // Fallback to original calculation
        $priceInfo = $this->customer_order_lines->colourways->price($this->sizes_id, $this->customer_order_lines->phase_id);
        if($priceInfo['status'] == 'success'){
            $price = $priceInfo['price'];
            return convertCurrency($price->gmt_trans_currency, '£', $price->gmt_trans, $this->customer_order_lines->customer_orders->seasons_id);
        }
    }

    public function getQuoteBaseAttribute(){
        $resolution = $this->getPriceResolution();
        if ($resolution) {
            return $resolution->quote_base;
        }

        // Fallback to original calculation
        $priceInfo = $this->customer_order_lines->colourways->price($this->sizes_id, $this->customer_order_lines->phase_id);
        if($priceInfo['status'] == 'success'){
            $price = $priceInfo['price'];
            return $price->quote_base;
        }
    }


    public function customer_order_lines()
    {
        return $this->belongsTo(CustomerOrderLines::class);
    }
    public function sizes()
    {
        return $this->belongsTo(Sizes::class);
    }

    /**
     * Get the price resolution for this order line quantity.
     */
    protected function getPriceResolution()
    {
        try {
            $service = app(\App\Services\PriceResolutionService::class);
            $colourway = $this->customer_order_lines->colourways;
            $seasonId = $this->customer_order_lines->customer_orders->seasons_id;
            
            return $service->resolve(
                $colourway->style_versions_id,
                $colourway->id,
                $this->sizes_id,
                $this->customer_order_lines->phase_id,
                $seasonId
            );
        } catch (\Exception $e) {
            \Log::error('Failed to get price resolution for customer order line quantity', [
                'customer_order_line_quantity_id' => $this->id,
                'error' => $e->getMessage()
            ]);
            return null;
        }
    }

}
