<?php

namespace App\Models;

use App\Casts\Currency;
use Illuminate\Support\Str;
use \Bkwld\Cloner\Cloneable;
use App\Models\BaseModel;
use OwenIt\Auditing\Contracts\Auditable;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class CustomerOrderLineQuantities extends BaseModel implements Auditable
{
    use \OwenIt\Auditing\Auditable;
    use SoftDeletes;
    use HasFactory;
    use Cloneable;

    protected $appends = ['price_model'];

    protected $casts = [
        'price' => Currency::class,
        'created_at' => 'datetime:Y-m-d',
        'updated_at' => 'datetime:Y-m-d',
        'deleted_at' => 'datetime:Y-m-d',
    ];
    protected $fillable = [
        'customer_order_lines_id',
        'sizes_id',
        'qty',
        'SKU',
        'barcode',
        'created_at',
        'updated_at',
        'price',
        'discount',
        'commission',
        'deleted_at',

        'quote_cache',
        'cmt_cache',
        'cost_cache',

        'prices_id',

        'transport_budget_cache',
        'quote_status_cache',
        'cmt_status_cache',
    ];

    public function getPriceModelAttribute(){
        // Use cached price resolution data only
        $resolution = $this->getPriceResolution();
        
        if ($resolution) {
            return [
                'quote_base' => $resolution->quote_base,
                'quote' => $resolution->quote,
                'discount_price' => $resolution->discount_price,
                'cmt' => $resolution->cmt,
                'subtotal' => $resolution->subtotal,
                'subtotal_base' => $resolution->subtotal_base,
                'transport_budget' => $resolution->gmt_trans_base,
                'transport_budget_base' => $resolution->gmt_trans_base,
                'quote_status' => $resolution->quote_status,
                'cmt_status' => $resolution->cmt_status,
                'yarn_cost_kg' => $resolution->yarn_value_euro,
            ];
        }

        // Return null if no cached resolution found - this will show "No cache" in the UI
        return null;
    }


    public function getTransportBudgetAttribute(){
        $resolution = $this->getPriceResolution();
        if ($resolution) {
            return $resolution->gmt_trans_base;
        }

        // Fallback to original calculation
        $priceInfo = $this->customer_order_lines->colourways->price($this->sizes_id, $this->customer_order_lines->phase_id);
        if($priceInfo['status'] == 'success'){
            $price = $priceInfo['price'];
            return $price->gmt_trans;
        }
    }

    public function getTransportBudgetBaseAttribute(){
        $resolution = $this->getPriceResolution();
        if ($resolution) {
            return $resolution->gmt_trans_base;
        }

        // Fallback to original calculation
        $priceInfo = $this->customer_order_lines->colourways->price($this->sizes_id, $this->customer_order_lines->phase_id);
        if($priceInfo['status'] == 'success'){
            $price = $priceInfo['price'];
            return convertCurrency($price->gmt_trans_currency, '£', $price->gmt_trans, $this->customer_order_lines->customer_orders->seasons_id);
        }
    }

    public function getQuoteBaseAttribute(){
        $resolution = $this->getPriceResolution();
        if ($resolution) {
            return $resolution->quote_base;
        }

        // Fallback to original calculation
        $priceInfo = $this->customer_order_lines->colourways->price($this->sizes_id, $this->customer_order_lines->phase_id);
        if($priceInfo['status'] == 'success'){
            $price = $priceInfo['price'];
            return $price->quote_base;
        }
    }


    public function customer_order_lines()
    {
        return $this->belongsTo(CustomerOrderLines::class);
    }
    public function sizes()
    {
        return $this->belongsTo(Sizes::class);
    }

    /**
     * Get the price resolution for this order line quantity.
     */
    protected function getPriceResolution()
    {
        // Check if we already have a cached resolution for this instance
        if (isset($this->_cached_price_resolution)) {
            return $this->_cached_price_resolution;
        }

        try {
            $service = app(\App\Services\PriceResolutionService::class);
            $colourway = $this->customer_order_lines->colourways;
            $seasonId = $this->customer_order_lines->customer_orders->seasons_id;
            
            $resolution = $service->resolve(
                $colourway->style_versions_id,
                $colourway->id,
                $this->sizes_id,
                $this->customer_order_lines->phase_id,
                $seasonId
            );
            
            // Cache the resolution for this instance
            $this->_cached_price_resolution = $resolution;
            
            return $resolution;
        } catch (\Exception $e) {
            \Log::error('Failed to get price resolution for customer order line quantity', [
                'customer_order_line_quantity_id' => $this->id,
                'error' => $e->getMessage()
            ]);
            $this->_cached_price_resolution = null;
            return null;
        }
    }

    /**
     * Set a cached price resolution for this instance.
     */
    public function setCachedPriceResolution($resolution)
    {
        $this->_cached_price_resolution = $resolution;
    }

    /**
     * Bulk load price resolutions for a collection of order line quantities.
     */
    public static function bulkLoadPriceResolutions($orderLineQuantities)
    {
        if ($orderLineQuantities->isEmpty()) {
            return;
        }

        // Build a query to get all price resolutions for these order line quantities
        $resolutionKeys = [];
        foreach ($orderLineQuantities as $qty) {
            $colourway = $qty->customer_order_lines->colourways;
            $seasonId = $qty->customer_order_lines->customer_orders->seasons_id;
            
            $resolutionKeys[] = [
                'style_versions_id' => $colourway->style_versions_id,
                'colourways_id' => $colourway->id,
                'sizes_id' => $qty->sizes_id,
                'phase_id' => $qty->customer_order_lines->phase_id,
                'season_id' => $seasonId,
            ];
        }

        // Get all price resolutions in one query
        $resolutions = \App\Models\PriceResolution::where(function($query) use ($resolutionKeys) {
            foreach ($resolutionKeys as $key) {
                $query->orWhere(function($q) use ($key) {
                    $q->where('style_versions_id', $key['style_versions_id'])
                      ->where('colourways_id', $key['colourways_id'])
                      ->where('sizes_id', $key['sizes_id'])
                      ->where('phase_id', $key['phase_id'])
                      ->where('season_id', $key['season_id'])
                      ->whereNotNull('fresh_at');
                });
            }
        })->get();

        // Create a lookup map for quick access
        $resolutionMap = [];
        foreach ($resolutions as $resolution) {
            $key = $resolution->style_versions_id . '|' . $resolution->colourways_id . '|' . $resolution->sizes_id . '|' . $resolution->phase_id . '|' . $resolution->season_id;
            $resolutionMap[$key] = $resolution;
        }

        // Assign resolutions to order line quantities
        foreach ($orderLineQuantities as $qty) {
            $colourway = $qty->customer_order_lines->colourways;
            $seasonId = $qty->customer_order_lines->customer_orders->seasons_id;
            
            $key = $colourway->style_versions_id . '|' . $colourway->id . '|' . $qty->sizes_id . '|' . $qty->customer_order_lines->phase_id . '|' . $seasonId;
            
            if (isset($resolutionMap[$key])) {
                $qty->setCachedPriceResolution($resolutionMap[$key]);
            } else {
                $qty->setCachedPriceResolution(null);
            }
        }
    }

}
