# Multi-Tenant Migration Guide

This guide explains how to migrate from the single-tenant system to the multi-tenant platform with subdomain support.

## 🎯 Overview

The multi-tenant system allows multiple photographers to have their own branded galleries:
- **Subdomains**: Each photographer gets `username.vybephoto.com`
- **Custom Domains**: Professional+ tiers can use their own domain (e.g., `johnsmithphotography.com`)
- **Isolated Data**: Each user's albums, photos, and settings are completely separate
- **Usage Tracking**: Automatic tracking of photo count and storage usage
- **Subscription Tiers**: Starter (£10/mo), Professional (£25/mo), Business (£75/mo)

## 📋 Prerequisites

1. PostgreSQL database (required for multi-tenancy)
2. Domain with wildcard DNS support (e.g., `*.vybephoto.com`)
3. Node.js 18+ with npm
4. (Optional) Google Cloud Storage for file storage

## 🚀 Migration Steps

### Step 1: Install Dependencies

```bash
npm install bcrypt
```

### Step 2: Run Database Migration

The migration script will:
- Create the `users` table with subscription management
- Add `user_id` to existing tables
- Create sessions table for authentication
- Add triggers for automatic usage tracking
- Create billing_events table for Stripe integration

```bash
# Connect to your PostgreSQL database
psql -U postgres -d photowebsite -f migrations/001-multi-tenant-schema.sql
```

### Step 3: Migrate Existing Data (Optional)

If you have existing albums/photos, you'll need to:

1. Create a default user account:

```sql
-- Create a default photographer account
INSERT INTO users (username, email, password_hash, full_name, subscription_tier, subscription_status)
VALUES (
  'demo-photographer',
  'photographer@example.com',
  '$2b$10$YourHashedPasswordHere',  -- Generate with bcrypt
  'Demo Photographer',
  'professional',
  'active'
) RETURNING id;
```

2. Assign existing data to this user:

```sql
-- Update all existing albums to belong to user ID 1 (adjust as needed)
UPDATE albums SET user_id = 1 WHERE user_id IS NULL;
UPDATE album_groups SET user_id = 1 WHERE user_id IS NULL;
```

3. Calculate initial usage for the user:

```sql
-- Update usage statistics for user
UPDATE users u
SET 
  current_photo_count = (
    SELECT COUNT(*) 
    FROM photos p 
    JOIN albums a ON p.album_id = a.id 
    WHERE a.user_id = u.id
  ),
  current_storage_bytes = (
    SELECT COALESCE(SUM(p.size_bytes), 0)
    FROM photos p
    JOIN albums a ON p.album_id = a.id
    WHERE a.user_id = u.id
  )
WHERE u.id = 1;
```

### Step 4: Configure Environment Variables

Add/update these environment variables:

```bash
# Base domain for multi-tenancy
BASE_DOMAIN=vybephoto.com

# Admin subdomain (for main marketing site)
ADMIN_SUBDOMAIN=www

# Database (PostgreSQL required)
USE_POSTGRES=true
DB_HOST=localhost
DB_PORT=5432
DB_NAME=photowebsite
DB_USER=postgres
DB_PASSWORD=your_password

# Cloud Storage (optional but recommended)
USE_CLOUD_STORAGE=true
GCS_BUCKET_NAME=vybephoto-uploads
GCS_PROJECT_ID=your-gcp-project

# Stripe (for subscriptions)
STRIPE_SECRET_KEY=sk_test_...
STRIPE_PUBLISHABLE_KEY=pk_test_...
STRIPE_WEBHOOK_SECRET=whsec_...

# Server
PORT=3000
NODE_ENV=production
```

### Step 5: Update Server Entry Point

You have two options:

**Option A: Replace the old server (recommended for production)**

```bash
# Backup old server
mv src/server.js src/server-single-tenant.js

# Use new multi-tenant server
mv src/server-multi-tenant.js src/server.js
```

**Option B: Run alongside (for testing)**

Update `package.json`:

```json
{
  "scripts": {
    "start": "node src/server.js",
    "start:multi": "node src/server-multi-tenant.js",
    "dev:multi": "nodemon src/server-multi-tenant.js"
  }
}
```

### Step 6: Configure DNS

Set up wildcard DNS for your domain:

```
Type: A or CNAME
Name: *
Value: Your server IP or domain
```

For custom domains, each photographer will need to:
1. Add a CNAME pointing to their subdomain
2. Verify ownership
3. Enable in their settings

### Step 7: Update Storage Module (Optional)

If using Cloud Storage, ensure the storage module supports multi-tenant paths:

```javascript
// User files are stored as: userId/albumId/filename
// Example: 123/456/photo.jpg
```

### Step 8: Start the Server

```bash
# Development
npm run dev:multi

# Production
npm start
```

## 🧪 Testing Multi-Tenancy

### 1. Register a Test User

Visit `http://localhost:3000/register` and create a user with username `test-photographer`.

### 2. Test Subdomain Access

For local development, add to `/etc/hosts`:

```
127.0.0.1 test-photographer.vybephoto.com
127.0.0.1 www.vybephoto.com
```

Then visit:
- `http://test-photographer.vybephoto.com:3000` - User's gallery
- `http://www.vybephoto.com:3000` - Main marketing site
- `http://localhost:3000/dashboard` - User dashboard

### 3. Test Usage Limits

Upload photos and verify:
- Photo count increases automatically
- Storage usage is tracked
- Limits are enforced based on subscription tier

## 📊 Subscription Tiers

| Tier | Price | Photos | Storage | Custom Domain |
|------|-------|--------|---------|---------------|
| Starter | £10/mo | 1,000 | 5 GB | No |
| Professional | £25/mo | 10,000 | 50 GB | Yes |
| Business | £75/mo | 30,000 | 150 GB | Yes |

All plans include:
- 14-day free trial
- Unlimited albums
- Unlimited visitors
- Custom branding (colors, logo)
- Age verification (optional)
- Private & public albums

## 🔐 Security Considerations

1. **Password Security**: Uses bcrypt with 10 rounds
2. **Session Management**: HTTP-only cookies with 30-day expiration
3. **Rate Limiting**: 
   - Authentication: 5 attempts per 15 minutes
   - Uploads: 500 per hour (configurable)
   - General: 200 requests per 15 minutes
4. **Data Isolation**: All queries filtered by `user_id`
5. **HTTPS Required**: In production, use HTTPS for all traffic

## 🎨 Customization

Each user can customize:
- Site name
- Primary color (affects buttons, links, etc.)
- Logo
- Contact email
- Social media links
- Hot Chocolate donation link
- Age verification requirement

## 📈 Usage Tracking

Usage is tracked automatically via database triggers:
- Photo uploads increment count and storage
- Photo deletions decrement count and storage
- Usage logs are stored in `usage_logs` table
- Real-time dashboard shows current usage

## 💳 Stripe Integration (Optional)

For billing functionality, implement:

1. Stripe Checkout for subscriptions
2. Webhook handlers for subscription events
3. Customer portal for managing payments
4. Usage-based billing alerts

See `src/server-multi-tenant.js` for webhook endpoint placeholders.

## 🐛 Troubleshooting

### Issue: Subdomain not working

**Solution**: Check DNS configuration and ensure wildcard DNS is set up correctly.

### Issue: User can't upload photos

**Solution**: Check usage limits in dashboard. User may need to upgrade plan.

### Issue: Login not working

**Solution**: Ensure bcrypt is installed and password was hashed correctly during registration.

### Issue: Photos not showing

**Solution**: Verify `user_id` is set correctly in albums table and storage paths include user ID.

## 📚 API Reference

### Key Functions

**User Management**
- `createUser(username, email, password, fullName)`
- `getUserByUsername(username)`
- `getUserByCustomDomain(domain)`
- `verifyUserPassword(email, password)`
- `updateUserProfile(userId, updates)`

**Authentication**
- `loginUser(email, password, req, res)`
- `logoutUser(req, res)`
- `registerUser(username, email, password, fullName)`
- `authenticateUser(req, res, next)` - Middleware
- `requireAuth(req, res, next)` - Middleware

**Tenant Detection**
- `detectTenant(req, res, next)` - Middleware
- `requireTenant(req, res, next)` - Middleware
- `getGalleryUrl(user)` - Helper

**Usage Tracking**
- `hasReachedPhotoLimit(userId)`
- `hasReachedStorageLimit(userId, additionalBytes)`
- `getUserUsageStats(userId)`

## 🚢 Production Deployment

1. **Database**: Use managed PostgreSQL (Cloud SQL, RDS, etc.)
2. **Storage**: Use Cloud Storage (GCS, S3) instead of local filesystem
3. **CDN**: Put CloudFlare or similar in front for performance
4. **SSL**: Enable HTTPS and set `secure: true` for cookies
5. **Environment**: Set `NODE_ENV=production`
6. **Monitoring**: Set up logging and error tracking
7. **Backups**: Regular database backups with point-in-time recovery

## 📝 Next Steps

1. ✅ Run database migration
2. ✅ Configure environment variables
3. ✅ Update DNS settings
4. ✅ Test with sample users
5. ⏳ Implement Stripe billing (optional)
6. ⏳ Add email notifications (optional)
7. ⏳ Set up monitoring and alerts
8. ⏳ Deploy to production

## 🤝 Support

For questions or issues:
1. Check the troubleshooting section above
2. Review the code comments in `src/db-multi-tenant.js` and `src/server-multi-tenant.js`
3. Test locally before deploying to production

---

**Note**: This is a significant architectural change. Test thoroughly before deploying to production with real users.


