<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class ReceiptGroup extends Model
{
    protected $fillable = [
        'name',
        'description',
        'grouping_method',
        'confidence_score',
        'grouping_reason',
        'user_id',
    ];

    protected $casts = [
        'confidence_score' => 'decimal:2',
    ];

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function receipts(): HasMany
    {
        return $this->hasMany(Receipt::class);
    }

    /**
     * Get the total amount for all receipts in this group
     */
    public function getTotalAmountAttribute(): float
    {
        return $this->receipts->sum('total_amount');
    }

    /**
     * Get the primary merchant name (most common or first)
     */
    public function getPrimaryMerchantAttribute(): ?string
    {
        $merchants = $this->receipts->pluck('merchant_name')->filter();
        if ($merchants->isEmpty()) {
            return null;
        }

        // Return the most common merchant name, or the first one if tied
        $counts = $merchants->countBy();
        return $counts->sortDesc()->keys()->first();
    }

    /**
     * Get the date range for receipts in this group
     */
    public function getDateRangeAttribute(): array
    {
        $dates = $this->receipts->pluck('receipt_date')->filter();
        if ($dates->isEmpty()) {
            return ['start' => null, 'end' => null];
        }

        return [
            'start' => $dates->min(),
            'end' => $dates->max(),
        ];
    }

    /**
     * Check if this group has any matched transactions
     */
    public function hasMatches(): bool
    {
        return $this->receipts->some(fn($receipt) => $receipt->matches()->exists());
    }

    /**
     * Get all transaction matches for receipts in this group
     */
    public function getTransactionMatches()
    {
        return $this->receipts->flatMap(fn($receipt) => $receipt->matches);
    }
}