<?php

namespace App\Jobs;

use App\Models\CustomRequest;
use App\Models\CustomSession;
use App\Services\Contracts\LLMService;
use App\Services\Contracts\TTSService;
use App\Services\Contracts\SafetyService;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Storage;

class GenerateCustomSessionJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public $queue = 'ai';

    public function __construct(public CustomRequest $requestModel)
    {
        $this->onQueue('ai');
        $this->backoff = [10, 60, 180];
    }

    public function handle(LLMService $llm, TTSService $tts, SafetyService $safety): void
    {
        $req = $this->requestModel->fresh();
        $req->update(['status' => 'generating']);

        $profile = $req->user->profile ?? null;
        $context = ['overrides' => $req->overrides, 'profile' => $profile];
        $prompt = json_encode($context);

        $decision = $safety->check($prompt, 'input');
        if ($decision->action === 'block') {
            $req->update(['status' => 'blocked']);
            return;
        }

        $script = $llm->generateScript($req->user, $context);
        $decisionOut = $safety->check($script->text, 'output');
        if ($decisionOut->action === 'block') {
            $req->update(['status' => 'blocked']);
            return;
        }

        $audio = $tts->synthesize($script->text, $profile?->voice_prefs ?? []);

        CustomSession::create([
            'request_id' => $req->id,
            'title' => $req->overrides['topic'] ?? 'Custom Session',
            'duration_sec' => $audio->durationSec,
            'audio_path' => $audio->storagePath,
            'transcript_path' => null,
            'model_meta' => $script->meta,
        ]);

        $req->update(['status' => 'completed']);
    }
}


