<?php

namespace App\Livewire;

use Livewire\Component;
use App\Models\Season;
use App\Models\Category;
use App\Models\Item;

class HomePage extends Component
{
    public function render()
    {
        $featuredSeasons = Season::orderBy('sort_order')
            ->limit(3)
            ->get();

        $featuredCategories = Category::orderBy('sort_order')
            ->limit(6)
            ->get();

        // Load a random image from an active item for each category
        $categoryImages = [];
        foreach ($featuredCategories as $category) {
            $categoryImages[$category->id] = $this->getRandomCategoryImage($category);
        }

        $featuredProducts = Item::with(['season', 'category', 'colourways' => function($q) {
                $q->orderBy('sort_order');
            }])
            ->latest()
            ->limit(8)
            ->get();

        return view('livewire.home-page', [
            'featuredSeasons' => $featuredSeasons,
            'featuredCategories' => $featuredCategories,
            'categoryImages' => $categoryImages,
            'featuredProducts' => $featuredProducts,
        ])->layout('layouts.shop');
    }

    private function getRandomCategoryImage(Category $category): ?string
    {
        // First try to find items with primary images (from images array)
        $itemsWithPrimaryImage = Item::where('categories_id', $category->id)
            ->where('is_active', true)
            ->whereNotNull('images')
            ->inRandomOrder()
            ->first();

        if ($itemsWithPrimaryImage && $itemsWithPrimaryImage->primary_image) {
            return $itemsWithPrimaryImage->primary_image;
        }

        // Otherwise, try to find items with colourway images
        $itemWithColourwayImage = Item::where('categories_id', $category->id)
            ->where('is_active', true)
            ->whereHas('colourways', function($q) {
                $q->where('is_active', true)
                  ->whereNotNull('image');
            })
            ->with(['colourways' => function($q) {
                $q->where('is_active', true)
                  ->whereNotNull('image')
                  ->limit(1);
            }])
            ->inRandomOrder()
            ->first();

        if ($itemWithColourwayImage && $itemWithColourwayImage->colourways->isNotEmpty()) {
            return $itemWithColourwayImage->colourways->first()->image;
        }

        return null;
    }
}

