<?php

namespace App\Services;

use App\Models\ReceiptLine;
use App\Models\VatAnalysis;
use App\Services\AI\VertexClient;
use App\Services\Matching\ClassificationService;
use Illuminate\Support\Facades\Log;

class CombinedLineAnalysisService
{
    public function __construct(
        private VertexClient $vertex,
        private ClassificationService $classification
    ) {}

    /**
     * Process a receipt line with combined classification and VAT analysis in a single API call
     * This reduces processing time from 2 API calls to 1 API call per line
     */
    public function processLine(ReceiptLine $line): void
    {
        try {
            $startTime = microtime(true);
            
            // Combined prompt for both classification and VAT analysis
            $model = config('vertex.models.classify', 'gemini-2.5-flash');
            $system = 'You are a UK business expense expert. For the given receipt line, determine:
1. Department and account classification
2. VAT rate and claimable VAT amount under UK VAT law

Output JSON with: department, account, is_personal (boolean), vat_rate, vat_amount_claimable, reasoning';
            
            $prompt = json_encode([
                'description' => $line->description,
                'line_total' => $line->line_total,
                'vat_rate_hint' => $line->vat_rate ?? 0,
                'merchant' => $line->receipt->merchant_name,
                'allowed_departments' => ClassificationService::allowedDepartments(),
                'allowed_accounts' => ClassificationService::allowedAccounts(),
            ]);
            
            $resp = $this->vertex->generate($model, $prompt, $system, ['responseMimeType' => 'application/json']);
            $json = $resp['json'];
            
            if (is_array($json)) {
                // Process classification results
                $this->processClassificationResults($line, $json);
                
                // Process VAT results
                $this->processVatResults($line, $json);
            }
            
            $endTime = microtime(true);
            $processingTime = round($endTime - $startTime, 2);
            
            Log::info("Combined line analysis completed", [
                'line_id' => $line->id,
                'processing_time' => $processingTime . 's'
            ]);
            
        } catch (\Exception $e) {
            Log::error('Combined line analysis failed', [
                'line_id' => $line->id,
                'error' => $e->getMessage()
            ]);
            
            // Fallback to separate processing if combined fails
            $this->fallbackProcessing($line);
        }
    }

    private function processClassificationResults(ReceiptLine $line, array $json): void
    {
        if (!empty($json['department'])) {
            $line->department_id = ClassificationService::mapDepartmentNameToId($json['department']) ?? $line->department_id;
        }
        if (!empty($json['account'])) {
            $line->account_id = ClassificationService::mapAccountNameToId($json['account']) ?? $line->account_id;
        }
        if (isset($json['is_personal'])) {
            $line->is_personal = (bool)$json['is_personal'];
        }
        $line->save();
    }

    private function processVatResults(ReceiptLine $line, array $json): void
    {
        $vatRate = $line->vat_rate ?? 20.0;
        $claimable = 0.0;
        $reason = 'Combined LLM analysis';

        if ($line->is_personal) {
            $claimable = 0.0;
        } else {
            if (isset($json['vat_rate'])) {
                $vatRate = (float) $json['vat_rate'];
            }
            if (isset($json['vat_amount_claimable'])) {
                $claimable = (float) $json['vat_amount_claimable'];
            }
            if (isset($json['reasoning'])) {
                $reason = $json['reasoning'];
            }

            // Fallback calculation if no claimable amount provided
            if (!$claimable) {
                if ($line->vat_amount !== null) {
                    $claimable = (float) $line->vat_amount;
                } else {
                    $claimable = round(($line->line_total * ($vatRate / (100 + $vatRate))), 2);
                }
            }
        }

        VatAnalysis::updateOrCreate(
            ['receipt_line_id' => $line->id],
            [
                'vat_rate' => $vatRate,
                'vat_amount_claimable' => $claimable,
                'reasoning' => $reason,
            ]
        );
    }

    private function fallbackProcessing(ReceiptLine $line): void
    {
        // Fallback to separate classification and VAT processing
        $this->classification->classifyReceiptLine($line);
        
        // Simple VAT calculation without API call
        $vatRate = $line->vat_rate ?? 20.0;
        $claimable = $line->is_personal ? 0.0 : round(($line->line_total * ($vatRate / (100 + $vatRate))), 2);
        
        VatAnalysis::updateOrCreate(
            ['receipt_line_id' => $line->id],
            [
                'vat_rate' => $vatRate,
                'vat_amount_claimable' => $claimable,
                'reasoning' => 'Fallback calculation',
            ]
        );
    }
}
