<?php

namespace App\Services;

use App\Models\ReceiptLine;
use App\Services\Matching\ClassificationService;
use App\Services\VAT\UkVatService;
use Illuminate\Support\Facades\Log;

class LineProcessingService
{
    public function __construct(
        private ClassificationService $classification,
        private UkVatService $vat,
    ) {}

    /**
     * Process a single receipt line with both classification and VAT analysis
     */
    public function processLine(ReceiptLine $line): void
    {
        try {
            // Process classification and VAT analysis in parallel using async
            $this->processLineAsync($line);
        } catch (\Exception $e) {
            Log::error('Line processing failed', [
                'line_id' => $line->id,
                'error' => $e->getMessage()
            ]);
        }
    }

    /**
     * Process multiple lines in parallel batches
     */
    public function processLinesInParallel(array $lines, int $batchSize = 3): void
    {
        $batches = array_chunk($lines, $batchSize);
        
        foreach ($batches as $batch) {
            $this->processBatch($batch);
        }
    }

    private function processBatch(array $lines): void
    {
        $promises = [];
        
        foreach ($lines as $line) {
            // Create async promises for each line
            $promises[] = $this->createAsyncPromise($line);
        }
        
        // Wait for all promises to complete
        $this->waitForPromises($promises);
    }

    private function createAsyncPromise(ReceiptLine $line): \GuzzleHttp\Promise\PromiseInterface
    {
        // Create a promise that will execute both classification and VAT analysis
        return \GuzzleHttp\Promise\coroutine(function () use ($line) {
            // Run classification and VAT analysis concurrently
            yield [
                'classification' => $this->classification->classifyReceiptLineAsync($line),
                'vat' => $this->vat->analyzeLineAsync($line)
            ];
        });
    }

    private function waitForPromises(array $promises): void
    {
        // Wait for all promises to settle
        \GuzzleHttp\Promise\settle($promises)->wait();
    }

    private function processLineAsync(ReceiptLine $line): void
    {
        // For now, process sequentially but with optimized batching
        $this->classification->classifyReceiptLine($line);
        $this->vat->analyzeLine($line);
    }
}
