-- Migration: Create promotional offers system
-- Allows creating special offers with extended trials, discounts, and custom limits

CREATE TABLE IF NOT EXISTS promotional_offers (
  id SERIAL PRIMARY KEY,
  
  -- Offer Identification
  offer_code TEXT UNIQUE NOT NULL,
  offer_name TEXT NOT NULL,
  description TEXT,
  
  -- Offer Type
  offer_type TEXT NOT NULL CHECK (offer_type IN ('extended_trial', 'discount', 'custom_tier', 'combo')),
  
  -- Target Tier (which tier this offer applies to)
  target_tier TEXT NOT NULL REFERENCES subscription_tiers(tier_name),
  
  -- Extended Trial Configuration
  trial_days_override INTEGER, -- NULL = use tier default
  
  -- Discount Configuration
  discount_type TEXT CHECK (discount_type IN ('percentage', 'fixed_amount', 'none')),
  discount_value INTEGER, -- percentage (e.g., 20 for 20%) or pence amount (e.g., 500 for £5.00)
  discount_duration_months INTEGER, -- how many months discount applies, NULL = forever
  
  -- Custom Tier Limits (overrides)
  custom_max_photos INTEGER, -- NULL = use tier default
  custom_max_storage_bytes BIGINT, -- NULL = use tier default
  custom_price_pence INTEGER, -- NULL = use tier default
  
  -- Stripe Integration
  stripe_coupon_id TEXT, -- Stripe coupon ID if discount is managed via Stripe
  stripe_promotion_code_id TEXT, -- Stripe promotion code ID
  
  -- Usage Limits
  max_redemptions INTEGER, -- NULL = unlimited
  current_redemptions INTEGER NOT NULL DEFAULT 0,
  max_redemptions_per_user INTEGER DEFAULT 1,
  
  -- Validity Period
  valid_from TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
  valid_until TIMESTAMP, -- NULL = no expiration
  
  -- Status
  is_active BOOLEAN NOT NULL DEFAULT TRUE,
  is_public BOOLEAN NOT NULL DEFAULT FALSE, -- if true, shows on public pricing page
  
  -- Internal Notes
  internal_notes TEXT, -- admin notes, not shown to users
  
  -- Tracking
  created_by INTEGER REFERENCES users(id),
  created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP
);

-- Redemption tracking table
CREATE TABLE IF NOT EXISTS offer_redemptions (
  id SERIAL PRIMARY KEY,
  offer_id INTEGER NOT NULL REFERENCES promotional_offers(id) ON DELETE CASCADE,
  user_id INTEGER NOT NULL REFERENCES users(id) ON DELETE CASCADE,
  
  -- Redemption details
  redeemed_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
  offer_details JSONB, -- snapshot of offer at time of redemption
  
  -- Stripe tracking
  stripe_subscription_id TEXT,
  stripe_coupon_applied TEXT,
  
  -- Status
  is_active BOOLEAN NOT NULL DEFAULT TRUE, -- false if subscription was cancelled/refunded
  
  UNIQUE(offer_id, user_id) -- prevent duplicate redemptions by same user
);

-- Indexes for performance
CREATE INDEX IF NOT EXISTS idx_promotional_offers_code ON promotional_offers(offer_code);
CREATE INDEX IF NOT EXISTS idx_promotional_offers_active ON promotional_offers(is_active, valid_until);
CREATE INDEX IF NOT EXISTS idx_promotional_offers_tier ON promotional_offers(target_tier);
CREATE INDEX IF NOT EXISTS idx_offer_redemptions_offer ON offer_redemptions(offer_id);
CREATE INDEX IF NOT EXISTS idx_offer_redemptions_user ON offer_redemptions(user_id);

-- Function to update updated_at timestamp
CREATE OR REPLACE FUNCTION update_promotional_offers_timestamp()
RETURNS TRIGGER AS $$
BEGIN
  NEW.updated_at = CURRENT_TIMESTAMP;
  RETURN NEW;
END;
$$ LANGUAGE plpgsql;

CREATE TRIGGER promotional_offers_updated_at
  BEFORE UPDATE ON promotional_offers
  FOR EACH ROW
  EXECUTE FUNCTION update_promotional_offers_timestamp();

-- Comments
COMMENT ON TABLE promotional_offers IS 'Promotional offers with extended trials, discounts, and custom configurations';
COMMENT ON COLUMN promotional_offers.offer_code IS 'Unique code users enter to redeem offer (e.g., SUMMER2025)';
COMMENT ON COLUMN promotional_offers.offer_type IS 'Type: extended_trial, discount, custom_tier, or combo';
COMMENT ON COLUMN promotional_offers.discount_type IS 'Discount type: percentage (e.g., 20 for 20% off) or fixed_amount (in pence)';
COMMENT ON COLUMN promotional_offers.discount_duration_months IS 'How many months discount applies (NULL = forever/until cancelled)';
COMMENT ON COLUMN promotional_offers.max_redemptions IS 'Maximum total redemptions allowed (NULL = unlimited)';
COMMENT ON COLUMN promotional_offers.is_public IS 'If true, offer appears on public pricing page';

COMMENT ON TABLE offer_redemptions IS 'Tracks which users have redeemed which offers';
COMMENT ON COLUMN offer_redemptions.offer_details IS 'JSON snapshot of offer configuration at time of redemption';

-- Sample offers (for demonstration)
INSERT INTO promotional_offers (
  offer_code, offer_name, description, offer_type, target_tier,
  trial_days_override, is_active, is_public, internal_notes
) VALUES 
  (
    'WELCOME30',
    '30-Day Free Trial',
    'Extended 30-day trial for new Starter users',
    'extended_trial',
    'starter',
    30,
    TRUE,
    TRUE,
    'Default extended trial offer for marketing campaigns'
  ),
  (
    'PRO50OFF',
    '50% Off Professional - First 3 Months',
    'Half price on Professional tier for first quarter',
    'discount',
    'professional',
    NULL,
    TRUE,
    FALSE,
    'Partner/referral discount'
  )
ON CONFLICT (offer_code) DO NOTHING;

-- Update the discount details for the PRO50OFF offer
UPDATE promotional_offers 
SET discount_type = 'percentage',
    discount_value = 50,
    discount_duration_months = 3
WHERE offer_code = 'PRO50OFF';

