<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Item;
use App\Models\Category;
use App\Models\Season;
use Illuminate\Http\Request;

class ItemController extends Controller
{
    public function index()
    {
        $items = Item::with(['category','season'])->orderBy('created_at','desc')->paginate(20);
        return view('admin.items.index', compact('items'));
    }

    public function bySeason(Season $season)
    {
        $items = Item::with(['category','season'])
            ->where('season_id', $season->id)
            ->orderBy('name')
            ->paginate(20);
        return view('admin.items.index', compact('items'))
            ->with('scopedSeason', $season);
    }

    public function create()
    {
        $categories = Category::orderBy('sort_order')->get();
        $seasons = Season::orderByDesc('year')->orderBy('half','desc')->get();
        return view('admin.items.create', compact('categories','seasons'));
    }

    public function createForSeason(Season $season)
    {
        $categories = Category::orderBy('sort_order')->get();
        $seasons = Season::orderByDesc('year')->orderBy('half','desc')->get();
        return view('admin.items.create', compact('categories','seasons'))
            ->with('prefillSeason', $season);
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'name' => 'required|string|max:200',
            'slug' => 'required|string|max:200|unique:items,slug',
            'sku' => 'required|string|max:80|unique:items,sku',
            'category_id' => 'required|exists:categories,id',
            'season_id' => 'nullable|exists:seasons,id',
            'description' => 'nullable|string',
            'colour_grid_columns' => 'nullable|integer|min:1|max:8',
            'has_colourways' => 'boolean',
            'is_active' => 'boolean',
        ]);
        $data['has_colourways'] = $request->boolean('has_colourways');
        $data['is_active'] = $request->boolean('is_active');
        Item::create($data);
        return redirect()->route('admin.items.index')->with('status','Item created');
    }

    public function edit(Item $item)
    {
        $categories = Category::orderBy('sort_order')->get();
        $seasons = Season::orderByDesc('year')->orderBy('half','desc')->get();
        $item->load('colourways');
        return view('admin.items.edit', compact('item','categories','seasons'));
    }

    public function update(Request $request, Item $item)
    {
        $data = $request->validate([
            'name' => 'required|string|max:200',
            'slug' => 'required|string|max:200|unique:items,slug,'.$item->id,
            'sku' => 'required|string|max:80|unique:items,sku,'.$item->id,
            'category_id' => 'required|exists:categories,id',
            'season_id' => 'nullable|exists:seasons,id',
            'description' => 'nullable|string',
            'colour_grid_columns' => 'nullable|integer|min:1|max:8',
            'has_colourways' => 'boolean',
            'is_active' => 'boolean',
        ]);
        $data['has_colourways'] = $request->boolean('has_colourways');
        $data['is_active'] = $request->boolean('is_active');
        // Optional attributes JSON
        if ($request->filled('attributes_json')) {
            $decoded = json_decode($request->input('attributes_json'), true);
            if (json_last_error() === JSON_ERROR_NONE) {
                $data['attributes'] = $decoded;
            }
        } elseif ($request->has('attributes_json') && $request->input('attributes_json') === '') {
            $data['attributes'] = null;
        }

        $item->update($data);

        // Colourways management
        // Update/delete existing
        foreach ((array) $request->input('colourways', []) as $cw) {
            if (!empty($cw['id'])) {
                $colourway = $item->colourways()->where('id', $cw['id'])->first();
                if ($colourway) {
                    if (!empty($cw['remove']) && (int)$cw['remove'] === 1) {
                        $colourway->delete();
                        continue;
                    }
                    $colourway->update([
                        'name' => $cw['name'] ?? $colourway->name,
                        'colour_code' => $cw['colour_code'] ?? $colourway->colour_code,
                        'image_path' => $cw['image_path'] ?? $colourway->image_path,
                        'sort_order' => isset($cw['sort_order']) ? (int)$cw['sort_order'] : $colourway->sort_order,
                        'is_active' => isset($cw['is_active']) ? (bool)$cw['is_active'] : $colourway->is_active,
                    ]);
                }
            }
        }

        // Create new colourways
        foreach ((array) $request->input('new_colourways', []) as $cw) {
            if (!empty($cw['name']) && !empty($cw['colour_code'])) {
                $item->colourways()->create([
                    'name' => $cw['name'],
                    'colour_code' => $cw['colour_code'],
                    'image_path' => $cw['image_path'] ?? null,
                    'sort_order' => isset($cw['sort_order']) ? (int)$cw['sort_order'] : 0,
                    'is_active' => isset($cw['is_active']) ? (bool)$cw['is_active'] : true,
                ]);
            }
        }
        return redirect()->route('admin.items.index')->with('status','Item updated');
    }
}


