<?php

namespace App\Services;

use App\Contracts\PlaybackService as PlaybackServiceContract;
use App\Models\PlaybackEvent;
use App\Models\Track;
use App\Models\User;
use Illuminate\Support\Facades\DB;

class PlaybackServiceImpl implements PlaybackServiceContract
{
    public function log(string $event, User $user, Track $track, ?\DateTimeInterface $at = null): PlaybackEvent
    {
        abort_unless(in_array($event, ['play', 'complete'], true), 422, 'Invalid event');

        return PlaybackEvent::create([
            'user_id' => $user->id,
            'track_id' => $track->id,
            'event' => $event,
            'at' => $at ? \Carbon\CarbonImmutable::parse($at) : now(),
        ]);
    }

    public function aggregatesForTrack(Track $track, ?\DateTimeInterface $from = null, ?\DateTimeInterface $to = null): array
    {
        return $this->aggregate('track_id', $track->id, $from, $to);
    }

    public function aggregatesForUser(User $user, ?\DateTimeInterface $from = null, ?\DateTimeInterface $to = null): array
    {
        return $this->aggregate('user_id', $user->id, $from, $to);
    }

    private function aggregate(string $column, $value, ?\DateTimeInterface $from, ?\DateTimeInterface $to): array
    {
        $query = PlaybackEvent::query()
            ->select('event', DB::raw('COUNT(*) as cnt'))
            ->where($column, $value)
            ->groupBy('event');

        if ($from) {
            $query->where('at', '>=', $from);
        }
        if ($to) {
            $query->where('at', '<=', $to);
        }

        $rows = $query->pluck('cnt', 'event');
        return [
            'play' => (int) ($rows['play'] ?? 0),
            'complete' => (int) ($rows['complete'] ?? 0),
        ];
    }
}


