<?php

namespace App\Jobs;

use App\Models\StatementTransaction;
use App\Services\AI\ExpenseClassificationAgent;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;

class ClassifyStatementTransaction implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public int $tries = 5;
    public int $timeout = 60;
    public int $maxExceptions = 3;

    public function backoff(): array
    {
        return [10, 30, 120, 300];
    }

    public function retryUntil(): \DateTimeInterface
    {
        return now()->addHours(6);
    }

    public function __construct(
        public StatementTransaction $transaction,
        public ?int $receiptId = null
    ) {}

    public function handle(ExpenseClassificationAgent $agent): void
    {
        try {
            // If no receipt is linked, we cannot classify yet.
            // Release the job back to the queue with a delay.
            if (!$this->receiptId) {
                $this->release(now()->addMinutes(15));
                return;
            }

            $receipt = \App\Models\Receipt::find($this->receiptId);

            // If receipt not found (e.g., deleted), fail the job.
            if (!$receipt) {
                $this->fail(new \Exception("Receipt with ID {$this->receiptId} not found."));
                return;
            }
            
            $result = $agent->classifyTransaction($this->transaction, $receipt);
            
            Log::info('Statement transaction classified', [
                'transaction_id' => $this->transaction->id,
                'merchant' => $this->transaction->merchant_name,
                'result' => $result,
            ]);

            // If classification suggests splitting and we have a receipt, handle it
            if (!empty($result['split_suggestion']) && $receipt) {
                $this->handleTransactionSplitting($agent, $receipt);
            }

        } catch (\Throwable $e) {
            Log::error('Failed to classify statement transaction', [
                'transaction_id' => $this->transaction->id,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);
            $schedule = (array) $this->backoff();
            $index = max(0, min((int)($this->attempts() - 1), count($schedule) - 1));
            $this->release((int) $schedule[$index]);
        }
    }

    /**
     * Handle transaction splitting based on receipt analysis
     */
    private function handleTransactionSplitting(ExpenseClassificationAgent $agent, \App\Models\Receipt $receipt): void
    {
        $splitAnalysis = $agent->shouldSplitTransaction($this->transaction, $receipt);
        
        if ($splitAnalysis['should_split'] && !empty($splitAnalysis['split_lines'])) {
            Log::info('Transaction splitting suggested', [
                'transaction_id' => $this->transaction->id,
                'reasoning' => $splitAnalysis['reasoning'],
                'split_lines' => $splitAnalysis['split_lines'],
            ]);

            // For now, just log the suggestion. In a full implementation,
            // you might want to create separate transaction records or
            // store the split information in the transaction's meta field
            $this->transaction->update([
                'meta' => array_merge($this->transaction->meta ?? [], [
                    'split_suggestion' => $splitAnalysis,
                    'split_analyzed_at' => now()->toISOString(),
                ])
            ]);
        }
    }

    /**
     * Handle job failure
     */
    public function failed(\Throwable $exception): void
    {
        Log::error('ClassifyStatementTransaction job failed', [
            'transaction_id' => $this->transaction->id,
            'error' => $exception->getMessage(),
        ]);
    }
}
