<?php

namespace App\Jobs;

use App\Models\StatementTransaction;
use App\Services\AI\ExpenseClassificationAgent;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;

class ClassifyStatementTransaction implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public function __construct(
        public StatementTransaction $transaction,
        public ?int $receiptId = null
    ) {}

    public function handle(ExpenseClassificationAgent $agent): void
    {
        try {
            $receipt = $this->receiptId ? \App\Models\Receipt::find($this->receiptId) : null;
            
            $result = $agent->classifyTransaction($this->transaction, $receipt);
            
            Log::info('Statement transaction classified', [
                'transaction_id' => $this->transaction->id,
                'merchant' => $this->transaction->merchant_name,
                'result' => $result,
            ]);

            // If classification suggests splitting and we have a receipt, handle it
            if (!empty($result['split_suggestion']) && $receipt) {
                $this->handleTransactionSplitting($agent, $receipt);
            }

        } catch (\Exception $e) {
            Log::error('Failed to classify statement transaction', [
                'transaction_id' => $this->transaction->id,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);
            
            // Don't fail the job, just log the error
            $this->fail($e);
        }
    }

    /**
     * Handle transaction splitting based on receipt analysis
     */
    private function handleTransactionSplitting(ExpenseClassificationAgent $agent, \App\Models\Receipt $receipt): void
    {
        $splitAnalysis = $agent->shouldSplitTransaction($this->transaction, $receipt);
        
        if ($splitAnalysis['should_split'] && !empty($splitAnalysis['split_lines'])) {
            Log::info('Transaction splitting suggested', [
                'transaction_id' => $this->transaction->id,
                'reasoning' => $splitAnalysis['reasoning'],
                'split_lines' => $splitAnalysis['split_lines'],
            ]);

            // For now, just log the suggestion. In a full implementation,
            // you might want to create separate transaction records or
            // store the split information in the transaction's meta field
            $this->transaction->update([
                'meta' => array_merge($this->transaction->meta ?? [], [
                    'split_suggestion' => $splitAnalysis,
                    'split_analyzed_at' => now()->toISOString(),
                ])
            ]);
        }
    }

    /**
     * Handle job failure
     */
    public function failed(\Throwable $exception): void
    {
        Log::error('ClassifyStatementTransaction job failed', [
            'transaction_id' => $this->transaction->id,
            'error' => $exception->getMessage(),
        ]);
    }
}

