<?php

namespace App\Http\Controllers;

use App\Models\Subscription;
use App\Models\Webhook;
use Illuminate\Http\Request;
use App\Services\IAP\AppleIAPService;

class IAPController extends Controller
{
    public function verify(Request $request, AppleIAPService $iap)
    {
        $data = $request->validate([
            'receipt' => ['required', 'string'],
        ]);

        // Call Apple and interpret result (placeholder logic until full implementation)
        $result = $iap->verifyReceipt($data['receipt']);
        $status = (int) ($result['status'] ?? 21000);
        if ($status !== 0) {
            return response()->json(['status' => 'invalid_receipt', 'apple_status' => $status], 422);
        }

        $productId = $result['product_id'] ?? 'premium_monthly';
        $expiresMs = (int) ($result['expires_date_ms'] ?? now()->addMonth()->getTimestampMs());
        $sub = Subscription::updateOrCreate([
            'user_id' => $request->user()->id,
        ], [
            'store' => 'apple',
            'product_id' => $productId,
            'status' => 'active',
            'period_end_at' => now()->createFromTimestampMs($expiresMs),
            'latest_receipt' => $result,
        ]);

        return response()->json(['status' => 'ok', 'subscription' => $sub]);
    }

    public function entitlements(Request $request)
    {
        $sub = Subscription::where('user_id', $request->user()->id)->whereIn('status', ['active', 'in_grace'])->first();

        return response()->json([
            'active' => (bool) $sub,
            'product_id' => $sub?->product_id,
            'period_end_at' => optional($sub?->period_end_at)->toISOString(),
        ]);
    }

    public function webhook(Request $request)
    {
        // TODO: Verify signature, prevent replay, enqueue processing
        $webhook = Webhook::firstOrCreate([
            'provider' => 'apple',
            'event' => $request->header('X-Event', 'unknown'),
            'payload' => $request->all(),
        ]);

        return response()->json(['id' => $webhook->id], 202);
    }
}
