<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration {
    public function up(): void
    {
        Schema::table('users', function (Blueprint $table) {
            if (!Schema::hasColumn('users', 'apple_sub_id')) {
                $table->string('apple_sub_id')->nullable()->unique();
            }
            if (!Schema::hasColumn('users', 'locale')) {
                $table->string('locale', 10)->default('en_GB');
            }
            if (!Schema::hasColumn('users', 'timezone')) {
                $table->string('timezone', 50)->default('Europe/London');
            }
            if (!Schema::hasColumn('users', 'tos_accepted_at')) {
                $table->timestampTz('tos_accepted_at')->nullable();
            }
            if (Schema::hasColumn('users', 'password')) {
                $table->renameColumn('password', 'password_hash');
            }
        });

        Schema::create('profiles', function (Blueprint $table) {
            $table->id();
            $table->foreignId('user_id')->constrained()->cascadeOnDelete();
            $table->string('display_name')->nullable();
            $table->jsonb('goals')->nullable();
            $table->jsonb('voice_prefs')->nullable();
            $table->text('sensitivity')->nullable();
            $table->timestampsTz();
            $table->unique('user_id');
        });

        Schema::create('consents', function (Blueprint $table) {
            $table->id();
            $table->foreignId('user_id')->constrained()->cascadeOnDelete();
            $table->string('type');
            $table->string('version');
            $table->timestampTz('accepted_at');
            $table->timestampsTz();
            $table->index(['user_id', 'type']);
        });

        Schema::create('subscriptions', function (Blueprint $table) {
            $table->id();
            $table->foreignId('user_id')->constrained()->cascadeOnDelete();
            $table->enum('store', ['apple']);
            $table->string('product_id');
            $table->enum('status', ['active', 'in_grace', 'expired', 'canceled']);
            $table->timestampTz('period_end_at')->nullable();
            $table->jsonb('latest_receipt')->nullable();
            $table->timestampsTz();
            $table->index(['user_id', 'status']);
        });

        Schema::create('library_categories', function (Blueprint $table) {
            $table->id();
            $table->string('name');
            $table->string('slug')->unique();
            $table->timestampsTz();
        });

        Schema::create('library_tracks', function (Blueprint $table) {
            $table->id();
            $table->foreignId('category_id')->constrained('library_categories')->cascadeOnDelete();
            $table->string('title');
            $table->string('slug')->unique();
            $table->unsignedInteger('duration_sec');
            $table->string('audio_path');
            $table->string('transcript_path')->nullable();
            $table->jsonb('tags')->nullable();
            $table->timestampTz('published_at')->nullable();
            $table->timestampsTz();
            $table->index(['category_id', 'published_at']);
        });

        Schema::create('custom_requests', function (Blueprint $table) {
            $table->id();
            $table->foreignId('user_id')->constrained()->cascadeOnDelete();
            $table->enum('source', ['onboarding', 'weekly', 'manual'])->default('manual');
            $table->jsonb('overrides')->nullable();
            $table->enum('status', ['pending', 'scheduled', 'generating', 'completed', 'blocked', 'failed'])->default('pending');
            $table->timestampTz('scheduled_for')->nullable();
            $table->timestampsTz();
            $table->index(['user_id', 'status']);
        });

        Schema::create('custom_sessions', function (Blueprint $table) {
            $table->id();
            $table->foreignId('request_id')->constrained('custom_requests')->cascadeOnDelete();
            $table->string('title')->nullable();
            $table->unsignedInteger('duration_sec');
            $table->string('audio_path');
            $table->string('transcript_path')->nullable();
            $table->jsonb('model_meta')->nullable();
            $table->timestampsTz();
        });

        Schema::create('safety_events', function (Blueprint $table) {
            $table->id();
            $table->foreignId('user_id')->nullable()->constrained()->nullOnDelete();
            $table->enum('context', ['input', 'output', 'iap', 'other']);
            $table->string('rule_code');
            $table->enum('action', ['allow', 'redact', 'block']);
            $table->jsonb('payload')->nullable();
            $table->timestampsTz();
            $table->index(['user_id', 'context']);
        });

        Schema::create('content_flags', function (Blueprint $table) {
            $table->id();
            $table->foreignId('user_id')->nullable()->constrained()->nullOnDelete();
            $table->string('item_type');
            $table->unsignedBigInteger('item_id');
            $table->string('reason')->nullable();
            $table->timestampsTz();
            $table->index(['item_type', 'item_id']);
        });

        Schema::create('webhooks', function (Blueprint $table) {
            $table->id();
            $table->string('provider');
            $table->string('event');
            $table->jsonb('payload');
            $table->timestampTz('processed_at')->nullable();
            $table->enum('status', ['pending', 'processed', 'failed'])->default('pending');
            $table->timestampsTz();
            $table->index(['provider', 'status']);
        });

        Schema::create('idempotency_keys', function (Blueprint $table) {
            $table->id();
            $table->string('key');
            $table->foreignId('user_id')->nullable()->constrained()->nullOnDelete();
            $table->string('method', 10);
            $table->string('path');
            $table->string('request_hash');
            $table->unsignedSmallInteger('response_code')->nullable();
            $table->longText('response_body')->nullable();
            $table->timestampsTz();
            $table->unique(['key', 'user_id', 'method', 'path']);
        });
    }

    public function down(): void
    {
        Schema::dropIfExists('idempotency_keys');
        Schema::dropIfExists('webhooks');
        Schema::dropIfExists('content_flags');
        Schema::dropIfExists('safety_events');
        Schema::dropIfExists('custom_sessions');
        Schema::dropIfExists('custom_requests');
        Schema::dropIfExists('library_tracks');
        Schema::dropIfExists('library_categories');
        Schema::dropIfExists('subscriptions');
        Schema::dropIfExists('consents');
        Schema::dropIfExists('profiles');

        Schema::table('users', function (Blueprint $table) {
            if (Schema::hasColumn('users', 'apple_sub_id')) {
                $table->dropColumn('apple_sub_id');
            }
            if (Schema::hasColumn('users', 'locale')) {
                $table->dropColumn('locale');
            }
            if (Schema::hasColumn('users', 'timezone')) {
                $table->dropColumn('timezone');
            }
            if (Schema::hasColumn('users', 'tos_accepted_at')) {
                $table->dropColumn('tos_accepted_at');
            }
            if (Schema::hasColumn('users', 'password_hash')) {
                $table->renameColumn('password_hash', 'password');
            }
        });
    }
};


