<?php

namespace App\Services;

use App\Models\Statement;
use App\Models\StatementTransaction;
use App\Jobs\ClassifyStatementTransaction;
use Illuminate\Support\Facades\Auth;
use Maatwebsite\Excel\Facades\Excel;
use PhpOffice\PhpSpreadsheet\IOFactory;

class StatementImportService
{
    public function import(
        string $path,
        string $cardholderName,
        string $accountNumberMasked,
        string $periodStart,
        string $periodEnd,
        ?string $originalFilename = null,
        ?string $storagePath = null,
    ): array {
        $statement = Statement::create([
            'cardholder_name' => $cardholderName,
            'account_number_masked' => $accountNumberMasked,
            'period_start' => $periodStart,
            'period_end' => $periodEnd,
            'uploaded_by_user_id' => Auth::id(),
            'source_filename' => $originalFilename,
            'storage_path' => $storagePath,
        ]);

        $spreadsheet = IOFactory::load($path);
        $sheet = $spreadsheet->getActiveSheet();
        $rows = $sheet->toArray(null, true, true, true);

        // Assume first row header; map columns by names
        $header = array_map(fn($v) => trim((string) $v), array_shift($rows));
        $colIndex = array_flip($header);

        $count = 0;
        $transactionIds = [];
        
        foreach ($rows as $row) {
            if (empty(array_filter($row))) {
                continue;
            }

            $transactionDate = self::cell($row, $colIndex, 'Transaction Date');
            $postedDate = self::cell($row, $colIndex, 'Posted Date');

            $transaction = StatementTransaction::create([
                'statement_id' => $statement->id,
                'cardholder_name' => self::cell($row, $colIndex, 'Cardholder Name') ?: $cardholderName,
                'account_number_masked' => self::cell($row, $colIndex, 'Account Number') ?: $accountNumberMasked,
                'transaction_date' => $this->normalizeDate($transactionDate),
                'posted_date' => $this->normalizeDate($postedDate),
                'transaction_time' => self::cell($row, $colIndex, 'Transaction Time') ?: null,
                'merchant_name' => self::cell($row, $colIndex, 'Merchant Name'),
                'amount' => self::cell($row, $colIndex, 'Amount'),
                'currency' => self::cell($row, $colIndex, 'Currency') ?: 'GBP',
                'original_amount' => self::cell($row, $colIndex, 'Original Amount') ?: null,
                'original_currency' => self::cell($row, $colIndex, 'Original Currency') ?: null,
                'conversion_rate' => self::cell($row, $colIndex, 'Conversion Rate') ?: null,
                'authorisation_code' => self::cell($row, $colIndex, 'Authorisation Code') ?: null,
                'transaction_id' => self::cell($row, $colIndex, 'Transaction ID') ?: null,
                'merchant_category' => self::cell($row, $colIndex, 'Merchant Category') ?: null,
                'transaction_type' => self::cell($row, $colIndex, 'Transaction Type') ?: null,
                'mcc_description' => self::cell($row, $colIndex, 'MCC Description') ?: null,
                'merchant_city' => self::cell($row, $colIndex, 'Merchant Town/City') ?: null,
                'merchant_state' => self::cell($row, $colIndex, 'Merchant County/State') ?: null,
                'merchant_postcode' => self::cell($row, $colIndex, 'Merchant Post code/Zipcode') ?: null,
                'mcc' => self::cell($row, $colIndex, 'MCC') ?: null,
                'statement_cycle' => self::cell($row, $colIndex, 'Statement Cycle') ?: null,
                'has_receipt' => strtolower((string) self::cell($row, $colIndex, 'Receipt?')) === 'yes',
                'user_id' => Auth::id(),
                'meta' => [
                    'details' => self::cell($row, $colIndex, 'Details \n(Travel | Subsistence | Entertainment etc)') ?: null,
                ],
            ]);
            
            $transactionIds[] = $transaction->id;
            $count++;
        }

        // Dispatch classification jobs for all imported transactions
        foreach ($transactionIds as $transactionId) {
            $transaction = StatementTransaction::find($transactionId);
            if ($transaction) {
                ClassifyStatementTransaction::dispatch($transaction)->delay(now()->addSeconds(5));
            }
        }

        return [
            'statement_id' => $statement->id,
            'transactions_imported' => $count,
        ];
    }

    private static function cell(array $row, array $colIndex, string $columnName): mixed
    {
        if (!isset($colIndex[$columnName])) {
            return null;
        }
        $index = $colIndex[$columnName];
        return $row[$index] ?? null;
    }

    private function normalizeDate($value): ?string
    {
        if (empty($value)) return null;
        
        // Try d/m/Y or spreadsheet formatted date serials
        if (is_numeric($value)) {
            $unix = (int) (($value - 25569) * 86400);
            return gmdate('Y-m-d', $unix);
        }
        
        $value = trim((string) $value);
        
        // Handle d/m/Y format
        $parts = explode('/', $value);
        if (count($parts) === 3) {
            return sprintf('%04d-%02d-%02d', (int)$parts[2], (int)$parts[1], (int)$parts[0]);
        }
        
        // Handle Y-m-d format
        if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $value)) {
            return $value;
        }
        
        // Handle d-m-Y format
        if (preg_match('/^(\d{1,2})-(\d{1,2})-(\d{4})$/', $value, $matches)) {
            return sprintf('%04d-%02d-%02d', (int)$matches[3], (int)$matches[2], (int)$matches[1]);
        }
        
        // Handle Excel date serial numbers that might be strings
        if (preg_match('/^\d+\.\d+$/', $value)) {
            $unix = (int) (($value - 25569) * 86400);
            return gmdate('Y-m-d', $unix);
        }
        
        // Try to parse with strtotime as a fallback
        $timestamp = strtotime($value);
        if ($timestamp !== false) {
            return date('Y-m-d', $timestamp);
        }
        
        // If all else fails, log the problematic value and return null
        \Log::warning('Unable to parse date value: ' . $value);
        return null;
    }
}


