<?php

namespace Tests\Unit;

use Tests\TestCase;
use App\Models\TotalCache;
use App\Models\ShipmentLine;
use App\Models\CustomerOrderLines;
use App\Models\CustomerOrderLineQuantities;
use App\Models\CustomerOrders;
use App\Models\Shipment;
use App\Models\ShipmentLineSizes;
use App\Models\Colourways;
use App\Models\Customer;
use App\Models\Suppliers;
use App\Models\StyleVersions;
use App\Models\Styles;
use App\Models\Designs;
use App\Models\Seasons;
use App\Models\Sizes;
use App\Models\Price;
use App\Services\TotalCacheService;
use Illuminate\Foundation\Testing\RefreshDatabase;

class TotalCacheServiceTest extends TestCase
{
    use RefreshDatabase;

    protected TotalCacheService $service;

    protected function setUp(): void
    {
        parent::setUp();
        $this->service = app(TotalCacheService::class);
    }

    /**
     * Test that set method stores data correctly
     */
    public function test_set_stores_data_correctly()
    {
        $data = ['total' => 100.50, 'count' => 5];
        
        $this->service->set('test_entity', 1, 'test_key', $data);
        
        $cached = TotalCache::where('entity_type', 'test_entity')
            ->where('entity_id', 1)
            ->where('cache_key', 'test_key')
            ->first();
            
        $this->assertNotNull($cached);
        $this->assertEquals($data, $cached->cached_data);
        $this->assertNotNull($cached->fresh_at);
    }

    /**
     * Test that get method retrieves fresh data
     */
    public function test_get_retrieves_fresh_data()
    {
        $data = ['total' => 200.75, 'count' => 10];
        
        $this->service->set('test_entity', 2, 'test_key', $data);
        
        $retrieved = $this->service->get('test_entity', 2, 'test_key');
        
        $this->assertEquals($data, $retrieved);
    }

    /**
     * Test that get method returns null for stale data
     */
    public function test_get_returns_null_for_stale_data()
    {
        $data = ['total' => 300.25, 'count' => 15];
        
        $this->service->set('test_entity', 3, 'test_key', $data);
        
        // Manually set fresh_at to null to simulate stale data
        TotalCache::where('entity_type', 'test_entity')
            ->where('entity_id', 3)
            ->where('cache_key', 'test_key')
            ->update(['fresh_at' => null]);
        
        $retrieved = $this->service->get('test_entity', 3, 'test_key');
        
        $this->assertNull($retrieved);
    }

    /**
     * Test that invalidateEntity marks cache as stale
     */
    public function test_invalidate_entity_marks_cache_stale()
    {
        $data = ['total' => 400.00, 'count' => 20];
        
        $this->service->set('test_entity', 4, 'test_key', $data);
        
        // Verify cache exists and is fresh
        $cached = TotalCache::where('entity_type', 'test_entity')
            ->where('entity_id', 4)
            ->where('cache_key', 'test_key')
            ->first();
        $this->assertNotNull($cached->fresh_at);
        
        // Invalidate entity
        $this->service->invalidateEntity('test_entity', 4);
        
        // Verify cache is marked as stale
        $cached->refresh();
        $this->assertNull($cached->fresh_at);
    }

    /**
     * Test that invalidateEntity with specific cache key marks only that key as stale
     */
    public function test_invalidate_entity_with_key_marks_specific_key_stale()
    {
        $data1 = ['total' => 500.00, 'count' => 25];
        $data2 = ['total' => 600.00, 'count' => 30];
        
        $this->service->set('test_entity', 5, 'key1', $data1);
        $this->service->set('test_entity', 5, 'key2', $data2);
        
        // Invalidate specific key
        $this->service->invalidateEntity('test_entity', 5, 'key1');
        
        // Verify only key1 is stale
        $key1 = TotalCache::where('entity_type', 'test_entity')
            ->where('entity_id', 5)
            ->where('cache_key', 'key1')
            ->first();
        $key2 = TotalCache::where('entity_type', 'test_entity')
            ->where('entity_id', 5)
            ->where('cache_key', 'key2')
            ->first();
            
        $this->assertNull($key1->fresh_at);
        $this->assertNotNull($key2->fresh_at);
    }

    /**
     * Test that invalidateEntities marks multiple entities as stale
     */
    public function test_invalidate_entities_marks_multiple_entities_stale()
    {
        $data = ['total' => 700.00, 'count' => 35];
        
        $this->service->set('test_entity', 6, 'test_key', $data);
        $this->service->set('test_entity', 7, 'test_key', $data);
        $this->service->set('test_entity', 8, 'test_key', $data);
        
        // Invalidate multiple entities
        $this->service->invalidateEntities('test_entity', [6, 7]);
        
        // Verify entities 6 and 7 are stale, 8 is still fresh
        $entity6 = TotalCache::where('entity_type', 'test_entity')
            ->where('entity_id', 6)
            ->where('cache_key', 'test_key')
            ->first();
        $entity7 = TotalCache::where('entity_type', 'test_entity')
            ->where('entity_id', 7)
            ->where('cache_key', 'test_key')
            ->first();
        $entity8 = TotalCache::where('entity_type', 'test_entity')
            ->where('entity_id', 8)
            ->where('cache_key', 'test_key')
            ->first();
            
        $this->assertNull($entity6->fresh_at);
        $this->assertNull($entity7->fresh_at);
        $this->assertNotNull($entity8->fresh_at);
    }

    /**
     * Test that invalidateCacheKey marks all entries with that key as stale
     */
    public function test_invalidate_cache_key_marks_all_entries_stale()
    {
        $data = ['total' => 800.00, 'count' => 40];
        
        $this->service->set('test_entity', 9, 'shared_key', $data);
        $this->service->set('other_entity', 10, 'shared_key', $data);
        $this->service->set('test_entity', 11, 'different_key', $data);
        
        // Invalidate shared key
        $this->service->invalidateCacheKey('shared_key');
        
        // Verify shared_key entries are stale, different_key is fresh
        $entity9 = TotalCache::where('entity_type', 'test_entity')
            ->where('entity_id', 9)
            ->where('cache_key', 'shared_key')
            ->first();
        $entity10 = TotalCache::where('entity_type', 'other_entity')
            ->where('entity_id', 10)
            ->where('cache_key', 'shared_key')
            ->first();
        $entity11 = TotalCache::where('entity_type', 'test_entity')
            ->where('entity_id', 11)
            ->where('cache_key', 'different_key')
            ->first();
            
        $this->assertNull($entity9->fresh_at);
        $this->assertNull($entity10->fresh_at);
        $this->assertNotNull($entity11->fresh_at);
    }

    /**
     * Test that warmupShipmentLine calculates and stores totals correctly
     */
    public function test_warmup_shipment_line_calculates_totals()
    {
        // Create test data
        $season = Seasons::factory()->create();
        $customer = Customer::factory()->create();
        $supplier = Suppliers::factory()->create();
        $design = Designs::factory()->create();
        $style = Styles::factory()->create([
            'customers_id' => $customer->id,
            'designs_id' => $design->id,
            'seasons_id' => $season->id,
        ]);
        $styleVersion = StyleVersions::factory()->create([
            'styles_id' => $style->id,
            'factory_id' => $supplier->id,
        ]);
        $colourway = Colourways::factory()->create([
            'style_versions_id' => $styleVersion->id,
        ]);
        $price = Price::factory()->create([
            'style_versions_id' => $styleVersion->id,
            'colourways_id' => $colourway->id,
            'cmt' => 10.50,
            'quote' => 15.75,
        ]);
        $customerOrder = CustomerOrders::factory()->create([
            'customers_id' => $customer->id,
        ]);
        $customerOrderLine = CustomerOrderLines::factory()->create([
            'customer_orders_id' => $customerOrder->id,
            'colourways_id' => $colourway->id,
        ]);
        $size = Sizes::factory()->create();
        $customerOrderLineQuantity = CustomerOrderLineQuantities::factory()->create([
            'customer_order_lines_id' => $customerOrderLine->id,
            'sizes_id' => $size->id,
            'qty' => 10,
        ]);
        $shipment = Shipment::factory()->create();
        $shipmentLine = ShipmentLine::factory()->create([
            'shipment_id' => $shipment->id,
            'customer_order_lines_id' => $customerOrderLine->id,
        ]);
        $shipmentLineSize = ShipmentLineSizes::factory()->create([
            'shipment_lines_id' => $shipmentLine->id,
            'sizes_id' => $size->id,
            'qty' => 5,
        ]);

        // Warm up shipment line
        $this->service->warmupShipmentLine($shipmentLine);

        // Check that cache was created
        $cached = $this->service->get('shipment_line', $shipmentLine->id, 'prices');
        $this->assertNotNull($cached);
        $this->assertArrayHasKey('quote', $cached);
        $this->assertArrayHasKey('cmt', $cached);
        $this->assertArrayHasKey('subtotal', $cached);
    }

    /**
     * Test that warmupCustomerOrderLine calculates and stores totals correctly
     */
    public function test_warmup_customer_order_line_calculates_totals()
    {
        // Create test data
        $season = Seasons::factory()->create();
        $customer = Customer::factory()->create();
        $supplier = Suppliers::factory()->create();
        $design = Designs::factory()->create();
        $style = Styles::factory()->create([
            'customers_id' => $customer->id,
            'designs_id' => $design->id,
            'seasons_id' => $season->id,
        ]);
        $styleVersion = StyleVersions::factory()->create([
            'styles_id' => $style->id,
            'factory_id' => $supplier->id,
        ]);
        $colourway = Colourways::factory()->create([
            'style_versions_id' => $styleVersion->id,
        ]);
        $price = Price::factory()->create([
            'style_versions_id' => $styleVersion->id,
            'colourways_id' => $colourway->id,
            'cmt' => 12.25,
            'quote' => 18.50,
        ]);
        $customerOrder = CustomerOrders::factory()->create([
            'customers_id' => $customer->id,
        ]);
        $customerOrderLine = CustomerOrderLines::factory()->create([
            'customer_orders_id' => $customerOrder->id,
            'colourways_id' => $colourway->id,
        ]);
        $size = Sizes::factory()->create();
        $customerOrderLineQuantity = CustomerOrderLineQuantities::factory()->create([
            'customer_order_lines_id' => $customerOrderLine->id,
            'sizes_id' => $size->id,
            'qty' => 8,
        ]);

        // Warm up customer order line
        $this->service->warmupCustomerOrderLine($customerOrderLine);

        // Check that cache was created
        $cached = $this->service->get('customer_order_line', $customerOrderLine->id, 'prices');
        $this->assertNotNull($cached);
        $this->assertArrayHasKey('quote', $cached);
        $this->assertArrayHasKey('cmt', $cached);
        $this->assertArrayHasKey('subtotal', $cached);
    }

    /**
     * Test that warmupCustomerOrder calculates and stores totals correctly
     */
    public function test_warmup_customer_order_calculates_totals()
    {
        // Create test data
        $season = Seasons::factory()->create();
        $customer = Customer::factory()->create();
        $supplier = Suppliers::factory()->create();
        $design = Designs::factory()->create();
        $style = Styles::factory()->create([
            'customers_id' => $customer->id,
            'designs_id' => $design->id,
            'seasons_id' => $season->id,
        ]);
        $styleVersion = StyleVersions::factory()->create([
            'styles_id' => $style->id,
            'factory_id' => $supplier->id,
        ]);
        $colourway = Colourways::factory()->create([
            'style_versions_id' => $styleVersion->id,
        ]);
        $price = Price::factory()->create([
            'style_versions_id' => $styleVersion->id,
            'colourways_id' => $colourway->id,
            'cmt' => 14.75,
            'quote' => 22.00,
        ]);
        $customerOrder = CustomerOrders::factory()->create([
            'customers_id' => $customer->id,
        ]);
        $customerOrderLine = CustomerOrderLines::factory()->create([
            'customer_orders_id' => $customerOrder->id,
            'colourways_id' => $colourway->id,
        ]);
        $size = Sizes::factory()->create();
        $customerOrderLineQuantity = CustomerOrderLineQuantities::factory()->create([
            'customer_order_lines_id' => $customerOrderLine->id,
            'sizes_id' => $size->id,
            'qty' => 12,
        ]);

        // Warm up customer order
        $this->service->warmupCustomerOrder($customerOrder);

        // Check that cache was created
        $cached = $this->service->get('customer_order', $customerOrder->id, 'prices');
        $this->assertNotNull($cached);
        $this->assertArrayHasKey('total_sale', $cached);
        $this->assertArrayHasKey('total_sale_base', $cached);
        $this->assertArrayHasKey('total_qty', $cached);
    }

    /**
     * Test that warmupShipment calculates and stores totals correctly
     */
    public function test_warmup_shipment_calculates_totals()
    {
        // Create test data
        $season = Seasons::factory()->create();
        $customer = Customer::factory()->create();
        $supplier = Suppliers::factory()->create();
        $design = Designs::factory()->create();
        $style = Styles::factory()->create([
            'customers_id' => $customer->id,
            'designs_id' => $design->id,
            'seasons_id' => $season->id,
        ]);
        $styleVersion = StyleVersions::factory()->create([
            'styles_id' => $style->id,
            'factory_id' => $supplier->id,
        ]);
        $colourway = Colourways::factory()->create([
            'style_versions_id' => $styleVersion->id,
        ]);
        $price = Price::factory()->create([
            'style_versions_id' => $styleVersion->id,
            'colourways_id' => $colourway->id,
            'cmt' => 16.25,
            'quote' => 24.50,
        ]);
        $customerOrder = CustomerOrders::factory()->create([
            'customers_id' => $customer->id,
        ]);
        $customerOrderLine = CustomerOrderLines::factory()->create([
            'customer_orders_id' => $customerOrder->id,
            'colourways_id' => $colourway->id,
        ]);
        $size = Sizes::factory()->create();
        $customerOrderLineQuantity = CustomerOrderLineQuantities::factory()->create([
            'customer_order_lines_id' => $customerOrderLine->id,
            'sizes_id' => $size->id,
            'qty' => 15,
        ]);
        $shipment = Shipment::factory()->create();
        $shipmentLine = ShipmentLine::factory()->create([
            'shipment_id' => $shipment->id,
            'customer_order_lines_id' => $customerOrderLine->id,
        ]);
        $shipmentLineSize = ShipmentLineSizes::factory()->create([
            'shipment_lines_id' => $shipmentLine->id,
            'sizes_id' => $size->id,
            'qty' => 10,
        ]);

        // Warm up shipment
        $this->service->warmupShipment($shipment);

        // Check that cache was created
        $cached = $this->service->get('shipment', $shipment->id, 'prices');
        $this->assertNotNull($cached);
        $this->assertArrayHasKey('goods_value', $cached);
        $this->assertArrayHasKey('transport_budget', $cached);
        $this->assertArrayHasKey('goods_qty', $cached);
        $this->assertArrayHasKey('cmt_total', $cached);
    }

    /**
     * Test that getCacheStats returns correct statistics
     */
    public function test_get_cache_stats_returns_correct_statistics()
    {
        // Create test data
        $data = ['total' => 1000.00, 'count' => 50];
        
        $this->service->set('entity1', 1, 'key1', $data);
        $this->service->set('entity1', 2, 'key1', $data);
        $this->service->set('entity2', 1, 'key1', $data);
        
        // Make one stale
        TotalCache::where('entity_type', 'entity1')
            ->where('entity_id', 1)
            ->where('cache_key', 'key1')
            ->update(['fresh_at' => null]);
        
        $stats = $this->service->getCacheStats();
        
        $this->assertEquals(3, $stats['total']);
        $this->assertEquals(2, $stats['fresh']);
        $this->assertEquals(1, $stats['stale']);
        $this->assertEquals(['entity1' => 2, 'entity2' => 1], $stats['by_entity']);
    }

    /**
     * Test that cleanupStale removes old cache entries
     */
    public function test_cleanup_stale_removes_old_cache_entries()
    {
        $data = ['total' => 1100.00, 'count' => 55];
        
        // Create fresh entry
        $this->service->set('test_entity', 1, 'fresh_key', $data);
        
        // Create stale entry
        $this->service->set('test_entity', 2, 'stale_key', $data);
        TotalCache::where('entity_type', 'test_entity')
            ->where('entity_id', 2)
            ->where('cache_key', 'stale_key')
            ->update(['fresh_at' => null, 'updated_at' => now()->subDays(35)]);
        
        // Cleanup stale entries
        $deletedCount = $this->service->cleanupStale(30);
        
        $this->assertEquals(1, $deletedCount);
        $this->assertEquals(1, TotalCache::count());
    }
}
