<?php

namespace App\Http\Livewire\Production\Reports;

use Auth;
use Carbon\Carbon;
use App\Models\Seasons;
use App\Models\Customer;
use App\Models\Suppliers;
use App\Models\YarnOrder;
use Illuminate\Support\Str;
use App\Models\ShipmentLine;
use App\Models\CustomerOrders;
use Livewire\Attributes\Locked;
use Livewire\Attributes\Computed;
use Livewire\Attributes\Validate;
use App\Http\Livewire\BaseComponent;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Response;

class FactoryOrders extends BaseComponent
{

    public $showPODropdown = false;

    public $hideFilters = false;

    #[Validate([
        'filter' => [
            'array',
            'season' => ['nullable', 'exists:seasons,id'],
            'factory' => ['nullable', 'exists:suppliers,id'],
            'customer' => ['nullable', 'exists:customers,id'],
            'po' => ['nullable', 'exists:customer_orders,customer_po'],
            'address' => ['nullable', 'string'],
            'created_since' => ['nullable', 'date_format:d-m-Y'],
        ],
        'hideFilters' => ['nullable', 'boolean']
    ])]
    public $filter = ['season' => "1", 'factory' => "1", 'customer' => "1", 'address' => "", 'created_since' => "01-01-2020"];
    public $po = [];

    #[Locked]
    public $contact;

    #[Locked]
    public $shipments;

    public function clearFilters(){
        $this->reset('filter');
    }

    #[Computed]
    public function seasons(){
        return Seasons::select('id', 'description', 'created_at')->get()->sortBy('created_at');
    }
    #[Computed]
    public function customers(){
        return Customer::get()->sortBy('name');
    }
    #[Computed]
    public function factories(){
        return Suppliers::with('supplier_addresses')->where('type', 'factory')->get()->sortBy('name');
    }


    public function mount(){

        $filter = $this->filters();
        if($filter != NULL)
            $this->filter = $filter;
        $this->getfilter();
        $this->contact['name'] = Auth::user()->name;
        $this->contact['email'] = Auth::user()->email;
        $this->contact['phone'] = Auth::user()->phone;
    }

    public function updatedFilter($value, $subVar){
        if($subVar == 'customer'){
            $this->filter['address'] = "";
            $this->filter['po'] = "";
        }
    }
    public function render()
    {
        Gate::authorize('shipment:read');
        $this->getfilter();
        $this->shipments = ShipmentLine::where(function($query) {
            if ( ! empty($this->filter['season'])) {
                $query->whereRelation('customer_order_lines.customer_orders', 'seasons_id', '=', $this->filter['season']);
            }
            if ( ! empty($this->filter['factory'])) {
                $query->whereRelation('customer_order_lines.colourways.style_versions', 'factory_id', '=', $this->filter['factory']);
            }
            if ( ! empty($this->filter['customer'])) {
                $query->whereRelation('customer_order_lines.customer_orders', 'customers_id', '=', $this->filter['customer']);
            }
            if ( ! empty($this->filter['address'])) {
                $query->whereRelation('customer_order_lines.customer_orders', 'customer_addresses_id', '=', $this->filter['address']);
            }
            if ( ! empty($this->po)) {
                $query->whereRelationIn('customer_order_lines', 'customer_orders_id', array_keys($this->po));
            }
            if ( ! empty($this->filter['created_since'])) {
                $query->whereRelation('customer_order_lines.customer_orders', 'created_at', '>=', Carbon::parse($this->filter['created_since']));
            }
        })->with([
            'shipments.transporters',
            'customer_order_lines.customer_order_line_quantities',
            'customer_order_lines.customer_orders',
            'customer_order_lines.yarn_order_line_styles.yarn_order_lines.yarn_orders',
            'customer_order_lines.colourways.style_versions.styles.designs',
            'customer_order_lines.customer_order_line_quantities.sizes',
            'customer_order_lines.colourways.samples',
            'customer_order_lines.colourways.style_versions.factories',
            'customer_order_lines.colourways.style_versions.prices',
            'customer_order_lines.colourways.colourway_yarns.yarn_colours.yarn.counts'
        ])
        ->get()->where('customer_order_lines.customer_orders.cancelled', FALSE);

        // dd($this->shipments->first()->customer_order_lines->customer_order_line_quantities->first()->price_model['cmt']);

        return view('livewire.production.reports.factory-orders');
    }

    public function totalValue(){
        $i = 0;
        foreach($this->shipments as $shipment){
            $i = $i + $shipment['total_prices']['cmt'];
        }
        return $i;
    }

    public function sizeTotals(){
        $totals = [];
        foreach($this->shipments as $shipment){
            foreach($shipment->shipment_line_sizes as $size){
                if(!empty($totals[$size->sizes->name]))
                    $totals[$size->sizes->name]['qty'] = $totals[$size->sizes->name]['qty'] + $size->qty;
                else
                    $totals[$size->sizes->name] = ['qty' => $size->qty, 'order' => $size->sizes->order];
            }
        }
        $totals = collect($totals)->sortBy('order');
        return $totals;
    }

    public function filters(){
        return auth()->user()->setting('report-factory-orders');
    }

    public function getfilter(){
        $filter = $this->filters();
        if($filter != NULL){
            if(!array_key_exists('created_since', $filter)){
                $this->reset('filter');
            }
            auth()->user()->settings(['report-factory-orders' => $this->filter]);
        }
        if($filter == NULL)
            auth()->user()->settings(['report-factory-orders' => []]);

        if($this->filter != $filter)
            auth()->user()->settings(['report-factory-orders' => $this->filter]);
    }
    public function getYarnOrders($custOrderLine){
        $prods = YarnOrder::whereRelation('yarn_order_lines.yarn_order_line_styles', 'customer_order_lines_id', $custOrderLine)->get();
        return $prods;
    }
    public function totalPrice($shipment){
        $val = 0;
        // dd($this->shipments->find($shipment)->customer_order_lines->customer_order_line_sizes);
        foreach($this->shipments->find($shipment)->customer_order_lines->customer_order_line_quantities as $cols){
            // Use price resolution system instead of old cache
            $priceModel = $cols->price_model;
            if ($priceModel && isset($priceModel['cmt'])) {
                $val = $val + ($cols->qty * $priceModel['cmt']);
            }
        }
        // dd($this->shipments->find($shipment)->customer_order_lines->customer_order_line_quantities);
        return $val;
        // $cmt = $this->shipments->find($shipment)->customer_order_lines->colourways->style_versions->cmt_if_confirmed();
        // if($qty > 0 && $cmt != 'Many' && $cmt > 0 && is_numeric($qty) && is_numeric($cmt)){
        //     return number_format((float)$qty * $cmt, 2, '.', '');
        // }
    }

    /**
     * Get CMT value for shipment using price resolution system
     */
    private function getCmtForShipment($shipment)
    {
        $priceModel = $shipment->customer_order_lines->price_model;
        if ($priceModel && isset($priceModel['cmt'])) {
            return $priceModel['cmt'];
        }
        
        return 0;
    }

    public function getPOs(){
        if(!empty($this->filter['customer'])){
            $query = CustomerOrders::
                where('seasons_id', $this->filter['season'])->
                where('customers_id', $this->filter['customer'])->
                whereRelation('customer_order_lines.colourways.style_versions', 'factory_id', '=', $this->filter['factory']);
            if(!empty($this->filter['address']))
                $query->whereRelation('customer_order_lines.customer_orders', 'customer_addresses_id', '=', $this->filter['address']);
            return $query->get();
        }
    }
    public function grandTotalQty(){
        $i = 0;
        foreach($this->shipments as $shipment){
            $i = $i + $shipment->shipment_line_sizes->sum('qty');
        }
        return $i;
    }
    public function grandTotalPrice(){
        $i = 0;
        foreach($this->shipments as $shipment){
            $i = $i + $this->totalPrice($shipment->id);
        }
        return $i;
    }
    public function exportCSV(){
        $headers = array(
            'Content-Type' => 'text/csv'
        );
        if (!File::exists(public_path()."/factoryorderexports")) {
            File::makeDirectory(public_path() . "/factoryorderexports");
        }
        $filename =  public_path("factoryorderexports/download" . Str::random(40), ".csv");
        $handle = fopen($filename, 'w');

        $sizes = [];
        foreach($this->shipments as $shipment){
            foreach($shipment->customer_order_lines->customer_order_line_quantities as $qty){
                // if(!array_search($qty->sizes->name, $sizes))
                    $sizes[] = [$qty->sizes->name, $qty->sizes->order];
            }
        }
        //$sizes = $this->shipments->groupBy('customer_order_lines.customer_order_line_quantities.sizes.name');
        $sizes = collect($sizes)->unique(0)->sortBy(1);

        $headings = [
            "Factory",
            "Issue Date",
            "RT No.",
            "PO No.",
            "Customer Ref",
            "Description",
            "Colour",
            "Yarn",
            "Size",
            "Total",
            "Price",
            "Value",
            "Yarn Order Ref",
            "ExFty",
        ];
        foreach($sizes as $size){
            $headings[] = $size[0];
        }

        fputcsv($handle, $headings);

        foreach ($this->shipments->sortBy('recentExFty') as $shipment) {
            fputcsv($handle, [
                $shipment->customer_order_lines->colourways->style_versions->factories->name,
                $shipment->customer_order_lines->customer_orders->order_date?->format('d-M'),
                $shipment->customer_order_lines->colourways->style_versions->styles->designs->id,
                $shipment->customer_order_lines->customer_orders->customer_po,
                $shipment->customer_order_lines->colourways->style_versions->styles->customer_ref,
                $shipment->customer_order_lines->colourways->style_versions->styles->designs->description,
                $shipment->customer_order_lines->colourways->name,
                'yarns',
                'sizes',
                $shipment->customer_order_lines->totalQty(),
                $this->getCmtForShipment($shipment),
                $this->totalPrice($shipment->id),
                'PRODS',
                $shipment->recentExFty?->format('d-M')
            ]);

        }

        fclose($handle);

        return Response::download($filename, "download.csv", $headers);
    }

    public function select($filter, $value = 0){
        if(is_string($this->$filter)){
            $this->$filter = [];
        }
        if($value == 0){
            $this->$filter = [];
        }
        else{
            if(($this->$filter[$value] ?? 0) == 0){
                $this->$filter[$value] = 1;
            }
            else{
                unset($this->$filter[$value]);
            }
        }

        // dd($this->po);

        // $this->updated($filter);

        // auth()->user()->settings(['wholesale_shipment_schedule_' . $filter => $this->$filter]);
    }
}
