<?php

namespace App\Livewire;

use App\Models\SafetyEvent;
use App\Models\User;
use Illuminate\Support\Facades\Auth;
use Livewire\Component;
use Livewire\Attributes\On;

class SafetyEventManagement extends Component
{
    public User $user;
    public $safetyEvents = [];
    public $showReportModal = false;
    public $showResolutionModal = false;
    public $eventToResolve = null;
    
    // Report form fields
    public $eventType = '';
    public $severity = 'low';
    public $description = '';
    public $metadata = [];

    protected $rules = [
        'eventType' => 'required|string|max:255',
        'severity' => 'required|in:low,medium,high,critical',
        'description' => 'required|string|min:10|max:1000',
    ];

    public function mount()
    {
        $this->user = Auth::user();
        $this->loadSafetyEvents();
    }

    public function loadSafetyEvents()
    {
        $this->safetyEvents = $this->user->safetyEvents()
            ->orderBy('created_at', 'desc')
            ->get();
    }

    public function openReportModal()
    {
        $this->showReportModal = true;
        $this->resetForm();
    }

    public function closeReportModal()
    {
        $this->showReportModal = false;
        $this->resetForm();
    }

    public function resetForm()
    {
        $this->eventType = '';
        $this->severity = 'low';
        $this->description = '';
        $this->metadata = [];
        $this->resetErrorBag();
    }

    public function reportSafetyEvent()
    {
        $this->validate();

        $safetyEvent = SafetyEvent::create([
            'user_id' => $this->user->id,
            'event_type' => $this->eventType,
            'severity' => $this->severity,
            'description' => $this->description,
            'metadata' => $this->metadata,
            'status' => 'open',
        ]);

        $this->loadSafetyEvents();
        $this->closeReportModal();

        $this->dispatch('safety-event-reported', [
            'message' => 'Safety event reported successfully. Our team will review it shortly.',
            'type' => 'success'
        ]);
    }

    public function openResolutionModal($eventId)
    {
        $this->eventToResolve = SafetyEvent::findOrFail($eventId);
        $this->showResolutionModal = true;
    }

    public function closeResolutionModal()
    {
        $this->showResolutionModal = false;
        $this->eventToResolve = null;
    }

    public function markAsResolved($eventId, $notes = null)
    {
        $event = SafetyEvent::findOrFail($eventId);
        
        if ($event->user_id !== $this->user->id) {
            $this->addError('event', 'You can only manage your own safety events.');
            return;
        }

        $event->markAsResolved($notes);
        $this->loadSafetyEvents();
        $this->closeResolutionModal();

        $this->dispatch('safety-event-updated', [
            'message' => 'Safety event marked as resolved.',
            'type' => 'success'
        ]);
    }

    public function getSeverityColor($severity)
    {
        return match($severity) {
            'low' => 'text-green-600 bg-green-100',
            'medium' => 'text-yellow-600 bg-yellow-100',
            'high' => 'text-orange-600 bg-orange-100',
            'critical' => 'text-red-600 bg-red-100',
            default => 'text-gray-600 bg-gray-100',
        };
    }

    public function getStatusColor($status)
    {
        return match($status) {
            'open' => 'text-blue-600 bg-blue-100',
            'in_progress' => 'text-yellow-600 bg-yellow-100',
            'resolved' => 'text-green-600 bg-green-100',
            'closed' => 'text-gray-600 bg-gray-100',
            default => 'text-gray-600 bg-gray-100',
        };
    }

    public function getEventTypeLabel($type)
    {
        return match($type) {
            'content_warning' => 'Content Warning',
            'technical_issue' => 'Technical Issue',
            'accessibility_concern' => 'Accessibility Concern',
            'privacy_concern' => 'Privacy Concern',
            'safety_concern' => 'Safety Concern',
            'other' => 'Other',
            default => ucfirst(str_replace('_', ' ', $type)),
        };
    }

    public function render()
    {
        return view('livewire.safety-event-management');
    }
}
