#!/bin/bash

# Setup script for creating the first superuser account
# Usage: ./setup-superuser.sh

echo "==================================="
echo "Vybe Photo - Superuser Setup"
echo "==================================="
echo ""

# Load environment variables if .env exists
if [ -f .env ]; then
  export $(cat .env | grep -v '^#' | xargs)
fi

# Default database connection values
DB_HOST=${DB_HOST:-localhost}
DB_PORT=${DB_PORT:-5432}
DB_NAME=${DB_NAME:-photowebsite}
DB_USER=${DB_USER:-postgres}

echo "Database Configuration:"
echo "  Host: $DB_HOST"
echo "  Port: $DB_PORT"
echo "  Database: $DB_NAME"
echo "  User: $DB_USER"
echo ""

# Check if psql is available
if ! command -v psql &> /dev/null; then
  echo "Error: psql (PostgreSQL client) is not installed or not in PATH"
  echo "Please install PostgreSQL client tools first"
  exit 1
fi

# Run the migration first
echo "Step 1: Running superuser migration..."
echo ""

if [ -f migrations/002-add-superuser-support.sql ]; then
  PGPASSWORD=$DB_PASSWORD psql -h "$DB_HOST" -p "$DB_PORT" -U "$DB_USER" -d "$DB_NAME" -f migrations/002-add-superuser-support.sql
  
  if [ $? -eq 0 ]; then
    echo "✓ Migration completed successfully"
  else
    echo "✗ Migration failed"
    exit 1
  fi
else
  echo "Warning: Migration file not found at migrations/002-add-superuser-support.sql"
  echo "The database may already have superuser support"
fi

echo ""
echo "Step 2: Creating superuser account..."
echo ""

# Prompt for user details
read -p "Enter username (lowercase, alphanumeric and hyphens only): " USERNAME
read -p "Enter email address: " EMAIL
read -sp "Enter password: " PASSWORD
echo ""
read -sp "Confirm password: " PASSWORD_CONFIRM
echo ""
echo ""

# Validate password match
if [ "$PASSWORD" != "$PASSWORD_CONFIRM" ]; then
  echo "Error: Passwords do not match"
  exit 1
fi

# Validate username format
if ! [[ "$USERNAME" =~ ^[a-z0-9-]{3,50}$ ]]; then
  echo "Error: Username must be 3-50 characters, lowercase letters, numbers, and hyphens only"
  exit 1
fi

# Validate email format (basic check)
if ! [[ "$EMAIL" =~ ^[a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,}$ ]]; then
  echo "Error: Invalid email format"
  exit 1
fi

# Generate password hash using Node.js
echo "Generating password hash..."
PASSWORD_HASH=$(node -e "
const bcrypt = require('bcrypt');
bcrypt.hash('$PASSWORD', 10, (err, hash) => {
  if (err) {
    console.error('Error generating hash');
    process.exit(1);
  }
  console.log(hash);
});
")

if [ -z "$PASSWORD_HASH" ]; then
  echo "Error: Failed to generate password hash"
  echo "Make sure bcrypt is installed: npm install bcrypt"
  exit 1
fi

# Create the superuser account
echo "Creating superuser account..."

PGPASSWORD=$DB_PASSWORD psql -h "$DB_HOST" -p "$DB_PORT" -U "$DB_USER" -d "$DB_NAME" <<EOF
INSERT INTO users (
  username, 
  email, 
  password_hash, 
  full_name,
  subscription_tier, 
  subscription_status,
  is_superuser,
  max_photos,
  max_storage_bytes
) VALUES (
  '$USERNAME',
  '$EMAIL',
  '$PASSWORD_HASH',
  'Superuser Administrator',
  'business',
  'active',
  TRUE,
  999999,
  999999999999
)
ON CONFLICT (username) DO UPDATE SET
  is_superuser = TRUE,
  subscription_status = 'active'
RETURNING id, username, email, is_superuser;
EOF

if [ $? -eq 0 ]; then
  echo ""
  echo "==================================="
  echo "✓ Superuser created successfully!"
  echo "==================================="
  echo ""
  echo "You can now log in at: http://app.localhost:3000/login"
  echo "Username: $USERNAME"
  echo "Email: $EMAIL"
  echo ""
  echo "After logging in, access the superuser portal at:"
  echo "http://app.localhost:3000/superuser"
  echo ""
else
  echo ""
  echo "✗ Failed to create superuser"
  echo "The user may already exist. Trying to update existing user..."
  echo ""
  
  # Try to update existing user to make them superuser
  PGPASSWORD=$DB_PASSWORD psql -h "$DB_HOST" -p "$DB_PORT" -U "$DB_USER" -d "$DB_NAME" <<EOF
UPDATE users 
SET is_superuser = TRUE,
    subscription_status = 'active',
    subscription_tier = 'business'
WHERE username = '$USERNAME' OR email = '$EMAIL'
RETURNING id, username, email, is_superuser;
EOF
  
  if [ $? -eq 0 ]; then
    echo ""
    echo "✓ Existing user updated to superuser status"
  else
    echo ""
    echo "✗ Failed to update user"
    exit 1
  fi
fi

echo ""
echo "Setup complete!"

