<?php

use Illuminate\Support\Facades\Route;
use Illuminate\Support\Facades\Storage;
use App\Http\Controllers\AdminController;

// Public routes
Route::view('/', 'welcome')->name('welcome');

// Media signed URL route
Route::get('/media/signed/{token}', function (string $token) {
    $mediaService = app(\App\Services\MediaService::class);
    $data = $mediaService->validateSignedUrlToken($token);
    
    if (!$data) {
        abort(404, 'Invalid or expired media URL');
    }
    
    $filePath = $data['file_path'];
    $disk = Storage::disk(config('filesystems.default'));
    
    if (!$disk->exists($filePath)) {
        abort(404, 'Media file not found');
    }
    
    return $disk->response($filePath);
})->name('media.signed');

// Library routes (public) - using Livewire components
Route::get('/library', \App\Livewire\LibraryPage::class)->name('library.index');
Route::get('/library/category/{slug}', \App\Livewire\LibraryPage::class)->name('library.category');
Route::get('/library/track/{slug}', \App\Livewire\TrackDetail::class)->name('library.track');
Route::get('/library/search', \App\Livewire\LibraryPage::class)->name('library.search');

// Custom session routes (public form, protected sessions) - using Livewire components
Route::get('/custom', \App\Livewire\CustomSessionForm::class)->name('custom.create');
Route::get('/custom/sessions', \App\Livewire\CustomSessionsList::class)
    ->middleware(['auth'])
    ->name('custom.sessions.index');
Route::get('/custom/sessions/{session}', \App\Livewire\CustomSessionsList::class)
    ->middleware(['auth'])
    ->name('custom.sessions.show');

// Protected routes
Route::middleware(['auth'])->group(function () {
    Route::view('dashboard', 'dashboard')->name('dashboard');
    Route::view('profile', 'profile')->name('profile');
    
    // Admin routes
    Route::middleware('admin')->group(function () {
        Route::get('/admin', [AdminController::class, 'index'])->name('admin.index');
        Route::get('/admin/categories', [AdminController::class, 'categories'])->name('admin.categories');
        Route::get('/admin/tracks', [AdminController::class, 'tracks'])->name('admin.tracks');
        Route::post('/admin/tracks', [AdminController::class, 'storeTrack'])->name('admin.tracks.store');
        Route::put('/admin/tracks/{id}', [AdminController::class, 'updateTrack'])->name('admin.tracks.update');
        Route::delete('/admin/tracks/{id}', [AdminController::class, 'destroyTrack'])->name('admin.tracks.destroy');
        Route::get('/admin/safety-events', [AdminController::class, 'safetyEvents'])->name('admin.safety-events');
        Route::get('/admin/custom-requests', [AdminController::class, 'customRequests'])->name('admin.custom-requests');
        Route::post('/admin/custom-requests/{id}/approve', [AdminController::class, 'approveRequest'])->name('admin.approve-request');
        Route::post('/admin/custom-requests/{id}/reject', [AdminController::class, 'rejectRequest'])->name('admin.reject-request');
        Route::post('/admin/safety-events/{id}/resolve', [AdminController::class, 'resolveSafetyEvent'])->name('admin.resolve-safety-event');
    });
});

require __DIR__.'/auth.php';
