<?php

use Illuminate\Support\Facades\Route;
use Illuminate\Support\Facades\Storage;
use App\Http\Controllers\AdminController;

// Public routes
Route::view('/', 'welcome')->name('welcome');

// Media signed URL route
Route::get('/media/signed/{token}', function (string $token) {
    $mediaService = app(\App\Services\MediaService::class);
    $data = $mediaService->validateSignedUrlToken($token);
    
    if (!$data) {
        abort(404, 'Invalid or expired media URL');
    }
    
    $filePath = $data['file_path'];
    $disk = Storage::disk(config('filesystems.default'));
    
    if (!$disk->exists($filePath)) {
        abort(404, 'Media file not found');
    }
    
    return $disk->response($filePath);
})->name('media.signed');

// Media streaming route with range request support
Route::get('/media/streaming/{token}', function (string $token) {
    $mediaService = app(\App\Services\MediaService::class);
    $data = $mediaService->validateStreamingUrlToken($token);
    
    if (!$data) {
        abort(404, 'Invalid or expired streaming URL');
    }
    
    $filePath = $data['file_path'];
    $disk = Storage::disk(config('filesystems.default'));
    
    if (!$disk->exists($filePath)) {
        abort(404, 'Media file not found');
    }
    
    // Get file size for range requests
    $fileSize = $disk->size($filePath);
    $mimeType = $mediaService->getAudioMetadata($filePath)['mime_type'] ?? 'audio/mpeg';
    
    // Handle range requests for streaming
    $range = request()->header('Range');
    if ($range && $mediaService->supportsRangeRequests(pathinfo($filePath, PATHINFO_EXTENSION))) {
        // Parse range header (e.g., "bytes=0-1023")
        if (preg_match('/bytes=(\d+)-(\d*)/', $range, $matches)) {
            $start = (int) $matches[1];
            $end = !empty($matches[2]) ? (int) $matches[2] : $fileSize - 1;
            
            if ($start >= $fileSize || $end >= $fileSize || $start > $end) {
                abort(416, 'Range Not Satisfiable');
            }
            
            $length = $end - $start + 1;
            $stream = $disk->readStream($filePath);
            
            // Seek to start position
            fseek($stream, $start);
            
            return response()->stream(function () use ($stream, $length) {
                $buffer = 8192; // 8KB chunks
                $remaining = $length;
                
                while ($remaining > 0 && !feof($stream)) {
                    $chunkSize = min($buffer, $remaining);
                    $chunk = fread($stream, $chunkSize);
                    echo $chunk;
                    $remaining -= strlen($chunk);
                    
                    if (connection_aborted()) {
                        break;
                    }
                }
                
                fclose($stream);
            }, 206, [
                'Content-Type' => $mimeType,
                'Content-Length' => $length,
                'Content-Range' => "bytes {$start}-{$end}/{$fileSize}",
                'Accept-Ranges' => 'bytes',
                'Cache-Control' => 'no-cache',
            ]);
        }
    }
    
    // Fallback to regular response for non-range requests
    return $disk->response($filePath, null, [
        'Content-Type' => $mimeType,
        'Accept-Ranges' => 'bytes',
        'Cache-Control' => 'no-cache',
    ]);
})->name('media.streaming');

// Library routes (public) - using Livewire components
Route::get('/library', \App\Livewire\LibraryPage::class)->name('library.index');
Route::get('/library/category/{slug}', \App\Livewire\LibraryPage::class)->name('library.category');
Route::get('/library/track/{slug}', \App\Livewire\TrackDetail::class)->name('library.track');
Route::get('/library/search', \App\Livewire\LibraryPage::class)->name('library.search');

// Custom session routes (public form, protected sessions) - using Livewire components
Route::get('/custom', \App\Livewire\CustomSessionForm::class)->name('custom.create');
Route::get('/custom/sessions', \App\Livewire\CustomSessionsList::class)
    ->middleware(['auth'])
    ->name('custom.sessions.index');
Route::get('/custom/sessions/{session}', \App\Livewire\CustomSessionsList::class)
    ->middleware(['auth'])
    ->name('custom.sessions.show');

// Protected routes
Route::middleware(['auth'])->group(function () {
    Route::view('dashboard', 'dashboard')->name('dashboard');
    Route::view('profile', 'profile')->name('profile');
    Route::get('/safety-consent', \App\Livewire\Pages\SafetyConsent::class)->name('safety-consent');
    
    // Admin routes
    Route::middleware('admin')->group(function () {
        Route::get('/admin', [AdminController::class, 'index'])->name('admin.index');
        Route::get('/admin/categories', [AdminController::class, 'categories'])->name('admin.categories');
        Route::get('/admin/tracks', [AdminController::class, 'tracks'])->name('admin.tracks');
        Route::post('/admin/tracks', [AdminController::class, 'storeTrack'])->name('admin.tracks.store');
        Route::put('/admin/tracks/{id}', [AdminController::class, 'updateTrack'])->name('admin.tracks.update');
        Route::delete('/admin/tracks/{id}', [AdminController::class, 'destroyTrack'])->name('admin.tracks.destroy');
        Route::get('/admin/safety-events', [AdminController::class, 'safetyEvents'])->name('admin.safety-events');
        Route::get('/admin/custom-requests', [AdminController::class, 'customRequests'])->name('admin.custom-requests');
        Route::post('/admin/custom-requests/{id}/approve', [AdminController::class, 'approveRequest'])->name('admin.approve-request');
        Route::post('/admin/custom-requests/{id}/reject', [AdminController::class, 'rejectRequest'])->name('admin.reject-request');
        Route::post('/admin/safety-events/{id}/resolve', [AdminController::class, 'resolveSafetyEvent'])->name('admin.resolve-safety-event');
    });
});

require __DIR__.'/auth.php';
