# 🔄 Routing Architecture Update

## Summary of Changes

The system now uses **three separate subdomains** instead of putting everything on the main domain:

### Old Architecture (Before)
```
www.vybephoto.com/
  ├── /                      → Login/Register page
  ├── /register              → Registration
  ├── /login                 → Login
  ├── /dashboard             → Photographer dashboard
  └── username.vybephoto.com → Client galleries
```

### New Architecture (After)
```
www.vybephoto.com            → Joomla marketing site (external)

app.vybephoto.com            → Photographer portal (Node.js)
  ├── /                      → Landing page (login/register/pricing)
  ├── /register              → Registration
  ├── /login                 → Login
  ├── /dashboard             → Dashboard (after login)
  └── /admin                 → Album management

username.vybephoto.com       → Client galleries (Node.js)
  ├── /                      → Gallery homepage
  ├── /a/:code               → View album
  └── /g/:code               → View group
```

## Why This Change?

1. **Separation of Concerns**: Marketing site (Joomla) separate from app portal (Node.js)
2. **Clearer User Experience**: Photographers know to go to `app.domain` to login
3. **Better Security**: Authentication routes isolated to one subdomain
4. **Scalability**: Main site can be handled by different infrastructure (Joomla)

## What Changed in the Code

### 1. Updated `src/tenant.js`
- Changed `ADMIN_SUBDOMAIN` from `www` to `app`
- Added `isAppSubdomain` flag to detect `app.vybephoto.com`
- App subdomain is treated as "no tenant" (like main site)

### 2. Updated `src/server-multi-tenant.js`
- Added `requireAppSubdomain()` middleware
- `/` route now has three behaviors:
  - On main domain → Shows Joomla placeholder message
  - On `app.domain` → Shows beautiful landing page with pricing
  - On `username.domain` → Shows photographer's gallery
- All auth routes (`/register`, `/login`, `/dashboard`, `/admin`) now require app subdomain

### 3. Updated Documentation
- `ROUTING-ARCHITECTURE.md` - Complete routing guide
- `MULTI-TENANT-SUMMARY.md` - Updated examples
- `setup-multi-tenant.sh` - Updated environment variables

## How to Test Locally

### 1. Update `/etc/hosts`

```bash
sudo nano /etc/hosts
```

Add these lines:
```
127.0.0.1 localhost
127.0.0.1 www.vybephoto.com
127.0.0.1 app.vybephoto.com
127.0.0.1 app.localhost
127.0.0.1 john-smith.vybephoto.com
127.0.0.1 john-smith.localhost
```

### 2. Set Environment Variable

```bash
export APP_SUBDOMAIN=app
export BASE_DOMAIN=vybephoto.com
```

Or create `.env`:
```bash
APP_SUBDOMAIN=app
BASE_DOMAIN=vybephoto.com
USE_POSTGRES=true
DB_HOST=localhost
DB_NAME=photowebsite
DB_USER=postgres
DB_PASSWORD=your_password
```

### 3. Start the Server

```bash
npm run start:multi
```

### 4. Test Each Domain

**Test 1: Main Domain (should show Joomla message)**
```bash
# Open in browser
http://www.vybephoto.com:3000
```
Expected: Message about Joomla with link to app portal

**Test 2: App Portal (landing page)**
```bash
# Open in browser
http://app.vybephoto.com:3000
# or
http://app.localhost:3000
```
Expected: Beautiful landing page with "Start Free 14-Day Trial" and pricing tiers

**Test 3: Register New User**
```bash
# Open in browser
http://app.vybephoto.com:3000/register
```
Expected: Registration form with username preview

**Test 4: After Registration**
```bash
# Open in browser (automatically redirected after signup)
http://app.vybephoto.com:3000/dashboard
```
Expected: Dashboard with usage stats

**Test 5: User Gallery**
```bash
# Open in browser (using username from registration)
http://your-username.vybephoto.com:3000
# or
http://your-username.localhost:3000
```
Expected: Your branded gallery homepage

**Test 6: Try to access auth routes from wrong subdomain**
```bash
# Should fail with 404
http://john-smith.vybephoto.com:3000/register
http://www.vybephoto.com:3000/dashboard
```
Expected: "This page is only available at app.vybephoto.com"

## Production Deployment

### DNS Configuration

You'll need two DNS records:

**For Joomla (main site):**
```
Type: A
Name: @ (or www)
Value: <Joomla server IP>
```

**For Node.js (app + galleries):**
```
Type: A
Name: app
Value: <Node.js server IP>

Type: A
Name: *
Value: <Node.js server IP>
```

Or use a single wildcard:
```
Type: A
Name: *
Value: <Node.js server IP>
```

Then configure reverse proxy to route `www` to Joomla.

### Nginx Example

```nginx
# Main site → Joomla
server {
    listen 80;
    server_name vybephoto.com www.vybephoto.com;
    
    location / {
        proxy_pass http://joomla_server:80;
        proxy_set_header Host $host;
    }
}

# App portal + galleries → Node.js
server {
    listen 80;
    server_name app.vybephoto.com *.vybephoto.com;
    
    location / {
        proxy_pass http://nodejs_server:3000;
        proxy_set_header Host $host;
        proxy_set_header X-Real-IP $remote_addr;
    }
}
```

## Troubleshooting

### Issue: "This page is only available at app.vybephoto.com"

**Cause**: Trying to access auth routes from wrong subdomain

**Solution**: Use `app.vybephoto.com` for login/register/dashboard

### Issue: Can't resolve app.localhost

**Cause**: Not in `/etc/hosts`

**Solution**: Add `127.0.0.1 app.localhost` to `/etc/hosts`

### Issue: Landing page not showing

**Cause**: Not accessing the app subdomain

**Solution**: Visit `http://app.vybephoto.com:3000` not `http://www.vybephoto.com:3000`

### Issue: Registration form shows but fails

**Cause**: Database not set up

**Solution**: Run migration: `psql -U postgres -d photowebsite -f migrations/001-multi-tenant-schema.sql`

## Migration from Old System

If you had code expecting auth routes on the main domain:

**Old URLs:**
```
www.vybephoto.com/register → BROKEN
www.vybephoto.com/login → BROKEN
www.vybephoto.com/dashboard → BROKEN
```

**New URLs:**
```
app.vybephoto.com/register → ✅ Works
app.vybephoto.com/login → ✅ Works
app.vybephoto.com/dashboard → ✅ Works
```

Update any hardcoded URLs or redirects to use `app.subdomain` instead.

## Summary

✅ **Main domain** (`www`) → Reserved for Joomla marketing site  
✅ **App subdomain** (`app`) → Login, register, dashboard, admin (Node.js)  
✅ **User subdomains** (`username`) → Individual galleries (Node.js)  
✅ **Custom domains** → Mapped to user subdomains (Node.js)  

All authentication and photographer management happens on `app.subdomain` only!


