<?php

namespace App\Console\Commands;

use App\Models\Receipt;
use App\Models\ReceiptGroup;
use App\Services\ReceiptProcessingService;
use Illuminate\Console\Command;

class TestReceiptPipeline extends Command
{
    protected $signature = 'test:receipt-pipeline {receipt_id? : Specific receipt ID to test}';
    protected $description = 'Test the new receipt processing pipeline with automatic grouping';

    public function handle(ReceiptProcessingService $processingService)
    {
        $this->info('Testing Receipt Processing Pipeline with Automatic Grouping...');
        
        $receiptId = $this->argument('receipt_id');
        
        if ($receiptId) {
            // Test specific receipt
            $receipt = Receipt::find($receiptId);
            if (!$receipt) {
                $this->error("Receipt {$receiptId} not found.");
                return;
            }
            
            $this->info("Testing receipt {$receiptId}: {$receipt->merchant_name}");
            $this->processReceipt($processingService, $receipt);
        } else {
            // Test all unmatched receipts
            $unmatchedReceipts = Receipt::whereNull('receipt_group_id')
                ->whereDoesntHave('matches')
                ->with(['lines'])
                ->take(5)
                ->get();

            if ($unmatchedReceipts->isEmpty()) {
                $this->warn('No unmatched receipts found to test.');
                return;
            }

            $this->info("Testing {$unmatchedReceipts->count()} unmatched receipts...");

            foreach ($unmatchedReceipts as $receipt) {
                $this->processReceipt($processingService, $receipt);
            }
        }

        // Show summary
        $this->showSummary();
    }

    private function processReceipt(ReceiptProcessingService $processingService, Receipt $receipt)
    {
        $this->line("Processing receipt {$receipt->id}: {$receipt->merchant_name} - £{$receipt->total_amount}");
        
        try {
            $processingService->process($receipt);
            
            // Check if receipt was grouped
            $receipt->refresh();
            if ($receipt->receipt_group_id) {
                $group = $receipt->receiptGroup;
                $this->line("  → Grouped into: {$group->name} ({$group->receipts->count()} receipts)");
                $this->line("  → Group total: £{$group->total_amount}");
            } else {
                $this->line("  → Not grouped (individual receipt)");
            }
            
            // Check if matched
            if ($receipt->matches()->exists()) {
                $this->line("  → Matched to transaction");
            } else {
                $this->line("  → Not matched");
            }
            
        } catch (\Exception $e) {
            $this->error("  → Error: " . $e->getMessage());
        }
        
        $this->line("");
    }

    private function showSummary()
    {
        $this->newLine();
        $this->info("Pipeline Test Summary:");
        
        $totalReceipts = Receipt::count();
        $groupedReceipts = Receipt::whereNotNull('receipt_group_id')->count();
        $matchedReceipts = Receipt::whereHas('matches')->count();
        $groups = ReceiptGroup::count();
        
        $this->line("Total receipts: {$totalReceipts}");
        $this->line("Grouped receipts: {$groupedReceipts}");
        $this->line("Matched receipts: {$matchedReceipts}");
        $this->line("Receipt groups: {$groups}");
        
        if ($groups > 0) {
            $this->newLine();
            $this->info("Receipt Groups:");
            $receiptGroups = ReceiptGroup::withCount('receipts')->get();
            foreach ($receiptGroups as $group) {
                $this->line("  - {$group->name} ({$group->receipts_count} receipts) - {$group->grouping_method}");
            }
        }
    }
}