<?php

namespace App\Console\Commands;

use App\Models\Receipt;
use App\Services\ReceiptProcessingService;
use Illuminate\Console\Command;

class TestReceiptPipeline extends Command
{
    protected $signature = 'test:receipt-pipeline {receipt_id? : Specific receipt ID to test}';
    protected $description = 'Test the receipt processing pipeline';

    public function handle(ReceiptProcessingService $processingService)
    {
        $this->info('Testing Receipt Processing Pipeline...');
        
        $receiptId = $this->argument('receipt_id');
        
        if ($receiptId) {
            // Test specific receipt
            $receipt = Receipt::find($receiptId);
            if (!$receipt) {
                $this->error("Receipt {$receiptId} not found.");
                return;
            }
            
            $this->info("Testing receipt {$receiptId}: {$receipt->merchant_name}");
            $this->processReceipt($processingService, $receipt);
        } else {
            // Test all unmatched receipts
            $unmatchedReceipts = Receipt::whereNull('receipt_group_id')
                ->whereDoesntHave('matches')
                ->with(['lines'])
                ->take(5)
                ->get();

            if ($unmatchedReceipts->isEmpty()) {
                $this->warn('No unmatched receipts found to test.');
                return;
            }

            $this->info("Testing {$unmatchedReceipts->count()} unmatched receipts...");

            foreach ($unmatchedReceipts as $receipt) {
                $this->processReceipt($processingService, $receipt);
            }
        }

        // Show summary
        $this->showSummary();
    }

    private function processReceipt(ReceiptProcessingService $processingService, Receipt $receipt)
    {
        $this->line("Processing receipt {$receipt->id}: {$receipt->merchant_name} - £{$receipt->total_amount}");
        
        try {
            $processingService->process($receipt);
            
            // Check if receipt was processed
            $receipt->refresh();
            $this->line("  → Processed (individual receipt)");
            
            // Check if matched
            if ($receipt->matches()->exists()) {
                $this->line("  → Matched to transaction");
            } else {
                $this->line("  → Not matched");
            }
            
        } catch (\Exception $e) {
            $this->error("  → Error: " . $e->getMessage());
        }
        
        $this->line("");
    }

    private function showSummary()
    {
        $this->newLine();
        $this->info("Pipeline Test Summary:");
        
        $totalReceipts = Receipt::count();
        $matchedReceipts = Receipt::whereHas('matches')->count();
        
        $this->line("Total receipts: {$totalReceipts}");
        $this->line("Matched receipts: {$matchedReceipts}");
    }
}