# Migration to PostgreSQL - Completed ✅

This project has been successfully migrated from SQLite to PostgreSQL as the primary database.

## Changes Made

### 1. Database Layer Updates

#### `src/db.js` - Simplified
- Removed all SQLite code and dependencies
- PostgreSQL is now the **default** database (`USE_POSTGRES=true` by default)
- JSON database available as optional fallback (`USE_JSON=true`)
- SQLite support completely removed

#### `src/db-postgres.js` - Enhanced
Added missing functions to match SQLite parity:
- `updateAlbumDescription()` - Update album descriptions
- `updateHotChocolateSettings()` - Configure hot chocolate donation button
- `setHomePageAlbum()` - Set an album as the home page
- `getHomePageAlbum()` - Get the home page album
- `updateAlbumSortOrder()` - Reorder albums
- `updatePhotoSortOrder()` - Reorder photos
- `getPhotoById()` - Get a single photo
- `deletePhoto()` - Delete a photo
- `deleteAlbum()` - Delete an album
- `deleteGroup()` - Delete a group with optional cascade
- `getSiteConfig()` - Get site configuration
- `updateSiteConfig()` - Update site configuration

Enhanced schema with new columns:
- `albums.description` - Album descriptions
- `albums.hot_choc_enabled` - Hot chocolate button toggle
- `albums.hot_choc_url` - Custom donation URL
- `albums.is_home_page_album` - Home page flag
- `albums.sort_order` - Custom album ordering
- `photos.sort_order` - Custom photo ordering
- `site_config` table - Site-wide configuration storage

### 2. Old Database Files Removed

Deleted SQLite database files:
- ✅ `data/db.sqlite`
- ✅ `data/db.sqlite-shm`
- ✅ `data/db.sqlite-wal`

### 3. Dependencies Updated

#### `package.json`
- ✅ Removed `better-sqlite3` dependency (no longer needed)
- ✅ Kept `pg` (PostgreSQL client) at version `^8.14.0`

### 4. Documentation Created/Updated

#### New Files:
- ✅ `POSTGRESQL-SETUP.md` - Complete PostgreSQL setup guide
- ✅ `setup-postgres.sh` - Interactive setup script
- ✅ `MIGRATION-TO-POSTGRES.md` - This file
- ✅ `.env.example` - Updated with PostgreSQL configuration (blocked by gitignore, template provided)

#### Updated Files:
- ✅ `README.md` - Updated Quick Start with PostgreSQL instructions
- ✅ `DEPLOYMENT.md` - Updated deployment guide with Cloud SQL setup

### 5. Configuration Changes

#### Environment Variables
PostgreSQL is now configured via environment variables:

```env
USE_POSTGRES=true          # PostgreSQL enabled by default
DB_HOST=localhost          # Database host
DB_PORT=5432               # Database port
DB_USER=postgres           # Database user
DB_PASSWORD=your_password  # Database password
DB_NAME=photowebsite       # Database name
```

For Google Cloud SQL (production):
```env
INSTANCE_UNIX_SOCKET=/cloudsql/PROJECT:REGION:INSTANCE
```

## Migration Steps for Existing Installations

If you're migrating from an existing SQLite installation:

### Local Development

1. **Install PostgreSQL**
   ```bash
   # macOS
   brew install postgresql@16
   brew services start postgresql@16
   
   # Ubuntu
   sudo apt install postgresql postgresql-contrib
   ```

2. **Run Setup Script**
   ```bash
   ./setup-postgres.sh
   ```
   
   Or manually:
   ```bash
   createdb photowebsite
   ```

3. **Configure Environment**
   ```bash
   cp .env.example .env
   # Edit .env with your PostgreSQL credentials
   ```

4. **Install Dependencies**
   ```bash
   npm install
   ```

5. **Start Application**
   ```bash
   npm start
   ```
   
   Tables will be created automatically on first run.

6. **Migrate Data (if needed)**
   If you have existing SQLite data, you'll need to export and import manually.

### Production Deployment (Google Cloud)

1. **Create Cloud SQL Instance**
   ```bash
   gcloud sql instances create photowebsite-db \
     --database-version=POSTGRES_15 \
     --tier=db-f1-micro \
     --region=europe-west1
   ```

2. **Set Password & Create Database**
   ```bash
   gcloud sql users set-password postgres \
     --instance=photowebsite-db \
     --password=SECURE_PASSWORD
   
   gcloud sql databases create photowebsite \
     --instance=photowebsite-db
   ```

3. **Store Secrets**
   ```bash
   echo -n "SECURE_PASSWORD" | \
     gcloud secrets create DB_PASSWORD \
       --data-file=- \
       --replication-policy=automatic
   ```

4. **Deploy with Cloud SQL Connection**
   ```bash
   gcloud run deploy nrw-web \
     --source . \
     --region=europe-west1 \
     --add-cloudsql-instances=PROJECT:REGION:photowebsite-db \
     --set-env-vars="USE_POSTGRES=true,DB_NAME=photowebsite,DB_USER=postgres,INSTANCE_UNIX_SOCKET=/cloudsql/PROJECT:REGION:photowebsite-db" \
     --set-secrets="DB_PASSWORD=DB_PASSWORD:latest"
   ```

## Database Schema

PostgreSQL automatically creates these tables:

### `album_groups`
- Organizes albums into groups
- Supports group codes for access

### `albums`
- Photo albums with metadata
- Includes public/private flags
- Custom descriptions and settings
- Hot chocolate donation button
- Home page album designation
- Custom sort ordering

### `photos`
- Individual photos
- Linked to albums via foreign key
- Custom sort ordering

### `site_config`
- Site-wide configuration
- Branding (colors, logos, names)
- Social media links
- Contact information
- Hot chocolate default settings

## Benefits of PostgreSQL

✅ **Better Performance** - Optimized for concurrent access  
✅ **Production Ready** - Battle-tested at scale  
✅ **Cloud Native** - Native Cloud SQL support  
✅ **ACID Compliance** - Data integrity guaranteed  
✅ **Rich Features** - JSON columns, full-text search, etc.  
✅ **Backup & Restore** - Enterprise-grade tools  
✅ **Connection Pooling** - Better resource management  

## Cost Impact

### Cloud SQL (Production)
- **db-f1-micro**: £15/month
- **Storage**: £0.17/GB/month
- **Backups**: £0.08/GB/month

Total added cost: ~£15-25/month for small deployments

### vs SQLite
- SQLite: No cost but not suitable for Cloud Run
- PostgreSQL: Small cost but production-ready and scalable

## Rollback (Not Recommended)

If you need to rollback (not recommended):

1. Restore `better-sqlite3` dependency
2. Revert `src/db.js` from git history
3. Set `USE_POSTGRES=false` and `USE_JSON=false`

However, the project is now fully PostgreSQL-focused and this migration is complete.

## Support

For PostgreSQL setup help:
- See `POSTGRESQL-SETUP.md`
- Run `./setup-postgres.sh` for interactive setup
- Check `DEPLOYMENT.md` for Cloud SQL deployment

## Status

✅ Migration Complete  
✅ All Functions Implemented  
✅ Documentation Updated  
✅ Old Files Removed  
✅ Dependencies Updated  
✅ Schema Enhanced  

Date: November 5, 2025

