<?php

namespace App\Models;

use App\Casts\TitleCase;
use \Bkwld\Cloner\Cloneable;
use App\Models\BaseModel;
use OwenIt\Auditing\Contracts\Auditable;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Factories\HasFactory;

/**
 * Material model
 * 
 * Represents yarn materials used in the system.
 */
class Materials extends BaseModel implements Auditable
{
    use \OwenIt\Auditing\Auditable;
    use SoftDeletes;
    use HasFactory;
    use Cloneable;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'material',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'material' => TitleCase::class,
        'created_at' => 'datetime:Y-m-d',
        'updated_at' => 'datetime:Y-m-d',
        'deleted_at' => 'datetime:Y-m-d',
    ];

    /**
     * The "booted" method of the model.
     * 
     * @return void
     */
    protected static function booted(): void
    {
        static::deleting(function ($model) { 
            $model->yarn_compositions?->each?->delete();
        });

        static::restoring(function ($model) { 
            $model->yarn_compositions()
                ->withTrashed()
                ->where('deleted_at', '>=', $model->deleted_at)
                ->each(function ($item, $key) { 
                    $item->restore(); 
                });
        });

        static::addGlobalScope('order', function (Builder $builder) {
            $builder->orderBy('material');
        });
    }

    /**
     * Get the yarn compositions for this material.
     * 
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function yarn_compositions()
    {
        return $this->hasMany(YarnComposition::class);
    }

    /**
     * Scope a query to search for materials.
     * 
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @param string $value
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeSearch($query, string $value)
    {
        return $query->where('material', 'like', "%{$value}%");
    }
}

