<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class CustomSession extends Model
{
    use HasFactory;

    protected $fillable = [
        'custom_request_id',
        'user_id',
        'title',
        'description',
        'transcript',
        'audio_path',
        'image_path',
        'duration_seconds',
        'status',
        'generation_metadata',
        'generated_at',
    ];

    protected $casts = [
        'generation_metadata' => 'array',
        'duration_seconds' => 'integer',
        'generated_at' => 'datetime',
    ];

    public function customRequest(): BelongsTo
    {
        return $this->belongsTo(CustomRequest::class);
    }

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function scopeByStatus($query, $status)
    {
        return $query->where('status', $status);
    }

    public function scopeByUser($query, $userId)
    {
        return $query->where('user_id', $userId);
    }

    public function isQueued(): bool
    {
        return $this->status === 'queued';
    }

    public function isGenerating(): bool
    {
        return $this->status === 'generating';
    }

    public function isReady(): bool
    {
        return $this->status === 'ready';
    }

    public function isFailed(): bool
    {
        return $this->status === 'failed';
    }

    public function getFormattedDurationAttribute(): string
    {
        $minutes = floor($this->duration_seconds / 60);
        $seconds = $this->duration_seconds % 60;
        return sprintf('%d:%02d', $minutes, $seconds);
    }

    public function markAsGenerating(): void
    {
        $this->update(['status' => 'generating']);
    }

    public function markAsReady(): void
    {
        $this->update([
            'status' => 'ready',
            'generated_at' => now(),
        ]);
    }

    public function markAsFailed(): void
    {
        $this->update(['status' => 'failed']);
    }
}
